<?php
/**
 * Attendance Controller
 */

class AttendanceController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $id, ?string $action, array $input): void {
        $user = AuthMiddleware::authenticate();

        // Handle special actions
        if ($action || $id) {
            switch ($id) {
                case 'check-in':
                    if ($method === 'POST') $this->checkIn($input);
                    return;
                case 'check-out':
                    if ($method === 'POST') $this->checkOut($input);
                    return;
                case 'break':
                    if ($method === 'POST') $this->startBreak($input);
                    return;
                case 'end-break':
                    if ($method === 'POST') $this->endBreak($input);
                    return;
                case 'today':
                    if ($method === 'GET') $this->getTodayAttendance();
                    return;
                case 'my':
                    if ($method === 'GET') $this->getMyAttendance();
                    return;
            }

            if ($action === 'vacation' && $method === 'POST') {
                $this->requestLeave($input);
                return;
            }
        }

        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getAttendance((int) $id);
                } else {
                    $this->listAttendance();
                }
                break;
            case 'PUT':
                AuthMiddleware::requireManager();
                $this->updateAttendance((int) $id, $input);
                break;
            default:
                Response::error('Method not allowed', 405);
        }
    }

    private function checkIn(array $input): void {
        $userId = AuthMiddleware::getUserId();
        $today = date('Y-m-d');
        $now = date('Y-m-d H:i:s');

        // Check if already checked in
        $existing = $this->db->fetchOne(
            "SELECT * FROM attendance WHERE user_id = ? AND date = ?",
            [$userId, $today]
        );

        if ($existing && $existing['check_in']) {
            Response::error('Already checked in today');
        }

        if ($existing) {
            // Update existing record
            $this->db->update('attendance', [
                'check_in' => $now,
                'check_in_gps_lat' => $input['gps_lat'] ?? null,
                'check_in_gps_lng' => $input['gps_lng'] ?? null,
                'check_in_ip' => AuditService::getClientIp(),
                'status' => 'present'
            ], 'id = ?', [$existing['id']]);
            $attendanceId = $existing['id'];
        } else {
            // Create new record
            $attendanceId = $this->db->insert('attendance', [
                'user_id' => $userId,
                'date' => $today,
                'check_in' => $now,
                'check_in_gps_lat' => $input['gps_lat'] ?? null,
                'check_in_gps_lng' => $input['gps_lng'] ?? null,
                'check_in_ip' => AuditService::getClientIp(),
                'status' => 'present'
            ]);
        }

        // Update user status
        $this->db->query(
            "INSERT INTO user_status (user_id, status, last_activity, gps_lat, gps_lng) 
             VALUES (?, 'online', NOW(), ?, ?) 
             ON DUPLICATE KEY UPDATE status = 'online', last_activity = NOW(), gps_lat = ?, gps_lng = ?",
            [$userId, $input['gps_lat'] ?? null, $input['gps_lng'] ?? null, $input['gps_lat'] ?? null, $input['gps_lng'] ?? null]
        );

        AuditService::log('check_in', 'attendance', $attendanceId);

        $attendance = $this->db->fetchOne("SELECT * FROM attendance WHERE id = ?", [$attendanceId]);
        Response::success($attendance, 'Checked in successfully');
    }

    private function checkOut(array $input): void {
        $userId = AuthMiddleware::getUserId();
        $today = date('Y-m-d');
        $now = date('Y-m-d H:i:s');

        $attendance = $this->db->fetchOne(
            "SELECT * FROM attendance WHERE user_id = ? AND date = ?",
            [$userId, $today]
        );

        if (!$attendance || !$attendance['check_in']) {
            Response::error('No check-in record found for today');
        }

        if ($attendance['check_out']) {
            Response::error('Already checked out today');
        }

        // End any active breaks
        $this->db->query(
            "UPDATE attendance_breaks SET break_end = ?, duration_minutes = TIMESTAMPDIFF(MINUTE, break_start, ?) 
             WHERE attendance_id = ? AND break_end IS NULL",
            [$now, $now, $attendance['id']]
        );

        // Calculate total break minutes
        $breakMinutes = $this->db->fetchOne(
            "SELECT COALESCE(SUM(duration_minutes), 0) as total FROM attendance_breaks WHERE attendance_id = ?",
            [$attendance['id']]
        );

        // Calculate total work minutes
        $checkIn = new DateTime($attendance['check_in']);
        $checkOut = new DateTime($now);
        $totalMinutes = ($checkOut->getTimestamp() - $checkIn->getTimestamp()) / 60;
        $workMinutes = max(0, $totalMinutes - (int)$breakMinutes['total']);

        $this->db->update('attendance', [
            'check_out' => $now,
            'check_out_gps_lat' => $input['gps_lat'] ?? null,
            'check_out_gps_lng' => $input['gps_lng'] ?? null,
            'check_out_ip' => AuditService::getClientIp(),
            'total_work_minutes' => (int) $workMinutes,
            'total_break_minutes' => (int) $breakMinutes['total']
        ], 'id = ?', [$attendance['id']]);

        // Update user status
        $this->db->query(
            "UPDATE user_status SET status = 'offline', current_task_id = NULL WHERE user_id = ?",
            [$userId]
        );

        AuditService::log('check_out', 'attendance', $attendance['id']);

        $updatedAttendance = $this->db->fetchOne("SELECT * FROM attendance WHERE id = ?", [$attendance['id']]);
        Response::success($updatedAttendance, 'Checked out successfully');
    }

    private function startBreak(array $input): void {
        $userId = AuthMiddleware::getUserId();
        $today = date('Y-m-d');
        $now = date('Y-m-d H:i:s');

        $attendance = $this->db->fetchOne(
            "SELECT * FROM attendance WHERE user_id = ? AND date = ? AND check_in IS NOT NULL AND check_out IS NULL",
            [$userId, $today]
        );

        if (!$attendance) {
            Response::error('No active attendance record found');
        }

        // Check for active break
        $activeBreak = $this->db->fetchOne(
            "SELECT * FROM attendance_breaks WHERE attendance_id = ? AND break_end IS NULL",
            [$attendance['id']]
        );

        if ($activeBreak) {
            Response::error('Already on break');
        }

        $breakId = $this->db->insert('attendance_breaks', [
            'attendance_id' => $attendance['id'],
            'break_start' => $now,
            'break_type' => $input['break_type'] ?? 'break'
        ]);

        // Update user status
        $this->db->query("UPDATE user_status SET status = 'away' WHERE user_id = ?", [$userId]);

        Response::success(['break_id' => $breakId], 'Break started');
    }

    private function endBreak(array $input): void {
        $userId = AuthMiddleware::getUserId();
        $today = date('Y-m-d');
        $now = date('Y-m-d H:i:s');

        $attendance = $this->db->fetchOne(
            "SELECT * FROM attendance WHERE user_id = ? AND date = ?",
            [$userId, $today]
        );

        if (!$attendance) {
            Response::error('No attendance record found');
        }

        $activeBreak = $this->db->fetchOne(
            "SELECT * FROM attendance_breaks WHERE attendance_id = ? AND break_end IS NULL",
            [$attendance['id']]
        );

        if (!$activeBreak) {
            Response::error('No active break found');
        }

        $breakStart = new DateTime($activeBreak['break_start']);
        $breakEnd = new DateTime($now);
        $durationMinutes = ($breakEnd->getTimestamp() - $breakStart->getTimestamp()) / 60;

        $this->db->update('attendance_breaks', [
            'break_end' => $now,
            'duration_minutes' => (int) $durationMinutes
        ], 'id = ?', [$activeBreak['id']]);

        // Update user status
        $this->db->query("UPDATE user_status SET status = 'online' WHERE user_id = ?", [$userId]);

        Response::success(['duration_minutes' => (int) $durationMinutes], 'Break ended');
    }

    private function getTodayAttendance(): void {
        $userId = AuthMiddleware::getUserId();
        $today = date('Y-m-d');

        $attendance = $this->db->fetchOne(
            "SELECT * FROM attendance WHERE user_id = ? AND date = ?",
            [$userId, $today]
        );

        if ($attendance) {
            // Get breaks
            $attendance['breaks'] = $this->db->fetchAll(
                "SELECT * FROM attendance_breaks WHERE attendance_id = ? ORDER BY break_start",
                [$attendance['id']]
            );

            // Check for active break
            $attendance['is_on_break'] = !empty(array_filter($attendance['breaks'], fn($b) => $b['break_end'] === null));
        }

        Response::success($attendance);
    }

    private function getMyAttendance(): void {
        $userId = AuthMiddleware::getUserId();
        $month = $_GET['month'] ?? date('Y-m');

        $attendance = $this->db->fetchAll(
            "SELECT * FROM attendance WHERE user_id = ? AND DATE_FORMAT(date, '%Y-%m') = ? ORDER BY date DESC",
            [$userId, $month]
        );

        // Get summary
        $summary = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total_days,
                SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
                SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_days,
                SUM(CASE WHEN status = 'vacation' THEN 1 ELSE 0 END) as vacation_days,
                SUM(CASE WHEN status = 'sick_leave' THEN 1 ELSE 0 END) as sick_leave_days,
                SUM(COALESCE(total_work_minutes, 0)) as total_work_minutes,
                SUM(COALESCE(total_break_minutes, 0)) as total_break_minutes
             FROM attendance 
             WHERE user_id = ? AND DATE_FORMAT(date, '%Y-%m') = ?",
            [$userId, $month]
        );

        Response::success([
            'records' => $attendance,
            'summary' => $summary
        ]);
    }

    private function listAttendance(): void {
        AuthMiddleware::requireManager();

        $page = (int) ($_GET['page'] ?? 1);
        $perPage = min((int) ($_GET['per_page'] ?? DEFAULT_PAGE_SIZE), MAX_PAGE_SIZE);
        $offset = ($page - 1) * $perPage;
        $date = $_GET['date'] ?? date('Y-m-d');

        $where = ['a.date = ?'];
        $params = [$date];

        if (!empty($_GET['user_id'])) {
            $where[] = 'a.user_id = ?';
            $params[] = $_GET['user_id'];
        }

        if (!empty($_GET['status'])) {
            $where[] = 'a.status = ?';
            $params[] = $_GET['status'];
        }

        $whereClause = implode(' AND ', $where);

        $attendance = $this->db->fetchAll(
            "SELECT a.*, u.name, u.employee_id 
             FROM attendance a 
             JOIN users u ON a.user_id = u.id 
             WHERE {$whereClause} 
             ORDER BY u.name 
             LIMIT ? OFFSET ?",
            array_merge($params, [$perPage, $offset])
        );

        $total = $this->db->fetchOne(
            "SELECT COUNT(*) as count FROM attendance a WHERE {$whereClause}",
            $params
        );

        Response::paginated($attendance, (int) $total['count'], $page, $perPage);
    }

    private function getAttendance(int $id): void {
        $attendance = $this->db->fetchOne(
            "SELECT a.*, u.name, u.employee_id 
             FROM attendance a 
             JOIN users u ON a.user_id = u.id 
             WHERE a.id = ?",
            [$id]
        );

        if (!$attendance) {
            Response::notFound('Attendance record not found');
        }

        $attendance['breaks'] = $this->db->fetchAll(
            "SELECT * FROM attendance_breaks WHERE attendance_id = ?",
            [$id]
        );

        Response::success($attendance);
    }

    private function updateAttendance(int $id, array $input): void {
        $attendance = $this->db->fetchOne("SELECT * FROM attendance WHERE id = ?", [$id]);

        if (!$attendance) {
            Response::notFound('Attendance record not found');
        }

        $updateData = [];
        $allowedFields = ['status', 'notes', 'check_in', 'check_out'];

        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateData[$field] = $input[$field];
            }
        }

        if (empty($updateData)) {
            Response::error('No valid fields to update');
        }

        $this->db->update('attendance', $updateData, 'id = ?', [$id]);

        AuditService::log('update', 'attendance', $id, $attendance, $updateData);

        $updated = $this->db->fetchOne("SELECT * FROM attendance WHERE id = ?", [$id]);
        Response::success($updated, 'Attendance updated');
    }

    private function requestLeave(array $input): void {
        $userId = AuthMiddleware::getUserId();

        $validator = new Validator($input);
        $validator
            ->required('date')
            ->date('date')
            ->required('status')
            ->in('status', ['vacation', 'sick_leave', 'home_leave'])
            ->validate();

        // Check if record exists
        $existing = $this->db->fetchOne(
            "SELECT * FROM attendance WHERE user_id = ? AND date = ?",
            [$userId, $input['date']]
        );

        if ($existing) {
            $this->db->update('attendance', [
                'status' => $input['status'],
                'notes' => $input['notes'] ?? null
            ], 'id = ?', [$existing['id']]);
            $attendanceId = $existing['id'];
        } else {
            $attendanceId = $this->db->insert('attendance', [
                'user_id' => $userId,
                'date' => $input['date'],
                'status' => $input['status'],
                'notes' => $input['notes'] ?? null
            ]);
        }

        AuditService::log('leave_request', 'attendance', $attendanceId);

        Response::success(null, 'Leave request submitted');
    }
}
