<?php
/**
 * Authentication Controller
 */

class AuthController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $action, array $input): void {
        switch ($action) {
            case 'login':
                if ($method === 'POST') $this->login($input);
                break;
            case 'register':
                if ($method === 'POST') $this->register($input);
                break;
            case 'refresh':
                if ($method === 'POST') $this->refreshToken($input);
                break;
            case 'logout':
                if ($method === 'POST') $this->logout();
                break;
            case 'me':
                if ($method === 'GET') $this->me();
                break;
            case 'change-password':
                if ($method === 'POST') $this->changePassword($input);
                break;
            case 'device-token':
                if ($method === 'POST') $this->registerDeviceToken($input);
                break;
            default:
                Response::notFound('Action not found');
        }
    }

    private function login(array $input): void {
        $validator = new Validator($input);
        $validator
            ->required('email')
            ->email('email')
            ->required('password')
            ->validate();

        $user = $this->db->fetchOne(
            "SELECT * FROM users WHERE email = ?",
            [$input['email']]
        );

        if (!$user || !password_verify($input['password'], $user['password'])) {
            Response::unauthorized('Invalid credentials');
        }

        if ($user['status'] !== 'active') {
            Response::forbidden('Account is ' . $user['status']);
        }

        // Update last login
        $this->db->update('users', ['last_login' => date('Y-m-d H:i:s')], 'id = ?', [$user['id']]);

        // Update user status
        $this->db->query(
            "INSERT INTO user_status (user_id, status, last_activity) 
             VALUES (?, 'online', NOW()) 
             ON DUPLICATE KEY UPDATE status = 'online', last_activity = NOW()",
            [$user['id']]
        );

        // Generate tokens
        $tokenData = [
            'user_id' => $user['id'],
            'email' => $user['email'],
            'role' => $user['role']
        ];
        $tokens = JWT::createTokenPair($tokenData);

        // Store refresh token
        $this->db->insert('refresh_tokens', [
            'user_id' => $user['id'],
            'token' => $tokens['refresh_token'],
            'expires_at' => date('Y-m-d H:i:s', time() + JWT_REFRESH_EXPIRY)
        ]);

        AuditService::log('login', 'user', $user['id']);

        unset($user['password']);
        
        Response::success([
            'user' => $user,
            'tokens' => $tokens
        ], 'Login successful');
    }

    private function register(array $input): void {
        // Only admin can register new users
        AuthMiddleware::requireAdmin();

        $validator = new Validator($input);
        $validator
            ->required('employee_id')
            ->required('name')
            ->required('email')
            ->email('email')
            ->required('password')
            ->minLength('password', 6)
            ->required('role')
            ->in('role', ['admin', 'manager', 'staff'])
            ->unique('email', 'users')
            ->unique('employee_id', 'users')
            ->validate();

        $userId = $this->db->insert('users', [
            'employee_id' => $input['employee_id'],
            'name' => $input['name'],
            'email' => $input['email'],
            'password' => password_hash($input['password'], PASSWORD_DEFAULT),
            'phone' => $input['phone'] ?? null,
            'role' => $input['role'],
            'department' => $input['department'] ?? null,
            'status' => 'active'
        ]);

        AuditService::log('create', 'user', $userId, null, $input);

        $user = $this->db->fetchOne("SELECT id, employee_id, name, email, role, status FROM users WHERE id = ?", [$userId]);

        Response::created($user, 'User registered successfully');
    }

    private function refreshToken(array $input): void {
        $validator = new Validator($input);
        $validator->required('refresh_token')->validate();

        $storedToken = $this->db->fetchOne(
            "SELECT * FROM refresh_tokens WHERE token = ? AND is_revoked = 0 AND expires_at > NOW()",
            [$input['refresh_token']]
        );

        if (!$storedToken) {
            Response::unauthorized('Invalid or expired refresh token');
        }

        $user = $this->db->fetchOne(
            "SELECT * FROM users WHERE id = ? AND status = 'active'",
            [$storedToken['user_id']]
        );

        if (!$user) {
            Response::unauthorized('User not found or inactive');
        }

        // Revoke old refresh token
        $this->db->update('refresh_tokens', ['is_revoked' => 1], 'id = ?', [$storedToken['id']]);

        // Generate new tokens
        $tokenData = [
            'user_id' => $user['id'],
            'email' => $user['email'],
            'role' => $user['role']
        ];
        $tokens = JWT::createTokenPair($tokenData);

        // Store new refresh token
        $this->db->insert('refresh_tokens', [
            'user_id' => $user['id'],
            'token' => $tokens['refresh_token'],
            'expires_at' => date('Y-m-d H:i:s', time() + JWT_REFRESH_EXPIRY)
        ]);

        Response::success(['tokens' => $tokens], 'Token refreshed');
    }

    private function logout(): void {
        $user = AuthMiddleware::authenticate();

        // Revoke all refresh tokens for this user
        $this->db->update('refresh_tokens', ['is_revoked' => 1], 'user_id = ?', [$user['id']]);

        // Update user status
        $this->db->query(
            "UPDATE user_status SET status = 'offline' WHERE user_id = ?",
            [$user['id']]
        );

        AuditService::log('logout', 'user', $user['id']);

        Response::success(null, 'Logged out successfully');
    }

    private function me(): void {
        $user = AuthMiddleware::authenticate();
        
        $fullUser = $this->db->fetchOne(
            "SELECT id, employee_id, name, email, phone, role, department, avatar, status, last_login, created_at 
             FROM users WHERE id = ?",
            [$user['id']]
        );

        // Get unread notification count
        $fullUser['unread_notifications'] = NotificationService::getUnreadCount($user['id']);

        Response::success($fullUser);
    }

    private function changePassword(array $input): void {
        $user = AuthMiddleware::authenticate();

        $validator = new Validator($input);
        $validator
            ->required('current_password')
            ->required('new_password')
            ->minLength('new_password', 6)
            ->validate();

        $fullUser = $this->db->fetchOne("SELECT password FROM users WHERE id = ?", [$user['id']]);

        if (!password_verify($input['current_password'], $fullUser['password'])) {
            Response::error('Current password is incorrect');
        }

        $this->db->update('users', [
            'password' => password_hash($input['new_password'], PASSWORD_DEFAULT)
        ], 'id = ?', [$user['id']]);

        // Revoke all refresh tokens
        $this->db->update('refresh_tokens', ['is_revoked' => 1], 'user_id = ?', [$user['id']]);

        AuditService::log('change_password', 'user', $user['id']);

        Response::success(null, 'Password changed successfully');
    }

    private function registerDeviceToken(array $input): void {
        $user = AuthMiddleware::authenticate();

        $validator = new Validator($input);
        $validator
            ->required('token')
            ->required('device_type')
            ->in('device_type', ['android', 'ios', 'web'])
            ->validate();

        // Deactivate existing token if exists
        $this->db->query(
            "UPDATE device_tokens SET is_active = 0 WHERE token = ?",
            [$input['token']]
        );

        // Insert new token
        $this->db->insert('device_tokens', [
            'user_id' => $user['id'],
            'token' => $input['token'],
            'device_type' => $input['device_type'],
            'device_name' => $input['device_name'] ?? null,
            'is_active' => 1
        ]);

        Response::success(null, 'Device token registered');
    }
}
