<?php
/**
 * Dashboard Controller
 */

class DashboardController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $action, array $input): void {
        $user = AuthMiddleware::authenticate();

        if ($method !== 'GET') {
            Response::error('Method not allowed', 405);
        }

        switch ($action) {
            case 'stats':
                $this->getStats();
                break;
            case 'my-stats':
                $this->getMyStats();
                break;
            case 'team-status':
                AuthMiddleware::requireManager();
                $this->getTeamStatus();
                break;
            case 'recent-tasks':
                $this->getRecentTasks();
                break;
            case 'sla-alerts':
                $this->getSlaAlerts();
                break;
            default:
                $this->getDashboard();
        }
    }

    private function getDashboard(): void {
        $user = AuthMiddleware::getCurrentUser();
        $userId = $user['id'];
        $today = date('Y-m-d');

        $data = [];

        // Today's attendance
        $data['attendance'] = $this->db->fetchOne(
            "SELECT * FROM attendance WHERE user_id = ? AND date = ?",
            [$userId, $today]
        );

        // My active tasks
        $data['active_tasks'] = $this->db->fetchAll(
            "SELECT t.* FROM tasks t 
             JOIN task_assignments ta ON t.id = ta.task_id 
             WHERE ta.user_id = ? AND t.status IN ('new', 'started', 'on_hold') 
             ORDER BY FIELD(t.priority, 'urgent', 'high', 'medium', 'low'), t.created_at DESC 
             LIMIT 5",
            [$userId]
        );

        // Task counts
        $data['task_counts'] = $this->db->fetchOne(
            "SELECT 
                SUM(CASE WHEN t.status = 'new' THEN 1 ELSE 0 END) as new_tasks,
                SUM(CASE WHEN t.status = 'started' THEN 1 ELSE 0 END) as in_progress,
                SUM(CASE WHEN t.status = 'on_hold' THEN 1 ELSE 0 END) as on_hold,
                SUM(CASE WHEN t.status = 'completed' AND DATE(t.end_time) = ? THEN 1 ELSE 0 END) as completed_today
             FROM tasks t 
             JOIN task_assignments ta ON t.id = ta.task_id 
             WHERE ta.user_id = ?",
            [$today, $userId]
        );

        // Unread notifications
        $data['unread_notifications'] = NotificationService::getUnreadCount($userId);

        // If manager, add team stats
        if (in_array($user['role'], ['admin', 'manager'])) {
            $data['team_stats'] = $this->getTeamStatsData();
        }

        Response::success($data);
    }

    private function getStats(): void {
        AuthMiddleware::requireManager();

        $today = date('Y-m-d');
        $thisMonth = date('Y-m');

        $stats = [];

        // Task statistics
        $stats['tasks'] = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'new' THEN 1 ELSE 0 END) as new_count,
                SUM(CASE WHEN status = 'started' THEN 1 ELSE 0 END) as started_count,
                SUM(CASE WHEN status = 'on_hold' THEN 1 ELSE 0 END) as on_hold_count,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_count,
                SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled_count,
                SUM(CASE WHEN sla_breached = 1 THEN 1 ELSE 0 END) as sla_breached_count
             FROM tasks"
        );

        // Today's statistics
        $stats['today'] = $this->db->fetchOne(
            "SELECT 
                SUM(CASE WHEN DATE(created_at) = ? THEN 1 ELSE 0 END) as created_today,
                SUM(CASE WHEN DATE(end_time) = ? AND status = 'completed' THEN 1 ELSE 0 END) as completed_today
             FROM tasks",
            [$today, $today]
        );

        // This month statistics
        $stats['this_month'] = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total_tasks,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_tasks,
                AVG(duration_minutes) as avg_duration_minutes
             FROM tasks 
             WHERE DATE_FORMAT(created_at, '%Y-%m') = ?",
            [$thisMonth]
        );

        // Tasks by type
        $stats['by_type'] = $this->db->fetchAll(
            "SELECT type, COUNT(*) as count FROM tasks GROUP BY type"
        );

        // Tasks by priority
        $stats['by_priority'] = $this->db->fetchAll(
            "SELECT priority, COUNT(*) as count FROM tasks WHERE status NOT IN ('completed', 'cancelled') GROUP BY priority"
        );

        // Attendance today
        $stats['attendance_today'] = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total_checked_in,
                (SELECT COUNT(*) FROM users WHERE status = 'active') as total_staff
             FROM attendance 
             WHERE date = ? AND check_in IS NOT NULL",
            [$today]
        );

        Response::success($stats);
    }

    private function getMyStats(): void {
        $userId = AuthMiddleware::getUserId();
        $thisMonth = date('Y-m');

        $stats = [];

        // My task counts
        $stats['tasks'] = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN t.status = 'new' THEN 1 ELSE 0 END) as new_count,
                SUM(CASE WHEN t.status = 'started' THEN 1 ELSE 0 END) as started_count,
                SUM(CASE WHEN t.status = 'completed' THEN 1 ELSE 0 END) as completed_count
             FROM tasks t 
             JOIN task_assignments ta ON t.id = ta.task_id 
             WHERE ta.user_id = ?",
            [$userId]
        );

        // This month performance
        $stats['this_month'] = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as tasks_assigned,
                SUM(CASE WHEN t.status = 'completed' THEN 1 ELSE 0 END) as tasks_completed,
                AVG(t.duration_minutes) as avg_duration_minutes,
                SUM(CASE WHEN t.sla_breached = 1 THEN 1 ELSE 0 END) as sla_breaches
             FROM tasks t 
             JOIN task_assignments ta ON t.id = ta.task_id 
             WHERE ta.user_id = ? AND DATE_FORMAT(t.created_at, '%Y-%m') = ?",
            [$userId, $thisMonth]
        );

        // Attendance this month
        $stats['attendance'] = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total_days,
                SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
                SUM(COALESCE(total_work_minutes, 0)) as total_work_minutes
             FROM attendance 
             WHERE user_id = ? AND DATE_FORMAT(date, '%Y-%m') = ?",
            [$userId, $thisMonth]
        );

        Response::success($stats);
    }

    private function getTeamStatus(): void {
        $users = $this->db->fetchAll(
            "SELECT u.id, u.name, u.employee_id, u.role,
                    us.status as live_status, us.last_activity, us.current_task_id,
                    t.task_id, t.title as current_task_title
             FROM users u
             LEFT JOIN user_status us ON u.id = us.user_id
             LEFT JOIN tasks t ON us.current_task_id = t.id
             WHERE u.status = 'active'
             ORDER BY 
                CASE us.status 
                    WHEN 'on_task' THEN 1 
                    WHEN 'online' THEN 2 
                    WHEN 'busy' THEN 3 
                    WHEN 'away' THEN 4 
                    ELSE 5 
                END,
                u.name"
        );

        Response::success($users);
    }

    private function getRecentTasks(): void {
        $limit = min((int) ($_GET['limit'] ?? 10), 50);

        $tasks = $this->db->fetchAll(
            "SELECT t.*, u.name as created_by_name
             FROM tasks t
             JOIN users u ON t.created_by = u.id
             ORDER BY t.created_at DESC
             LIMIT ?",
            [$limit]
        );

        Response::success($tasks);
    }

    private function getSlaAlerts(): void {
        $tasks = $this->db->fetchAll(
            "SELECT t.*, u.name as created_by_name,
                    TIMESTAMPDIFF(MINUTE, NOW(), t.sla_deadline) as minutes_remaining
             FROM tasks t
             JOIN users u ON t.created_by = u.id
             WHERE t.status NOT IN ('completed', 'cancelled')
               AND t.sla_deadline IS NOT NULL
               AND t.sla_deadline <= DATE_ADD(NOW(), INTERVAL 2 HOUR)
             ORDER BY t.sla_deadline ASC
             LIMIT 20"
        );

        Response::success($tasks);
    }

    private function getTeamStatsData(): array {
        $today = date('Y-m-d');

        return [
            'online_count' => (int) $this->db->fetchOne(
                "SELECT COUNT(*) as count FROM user_status WHERE status IN ('online', 'on_task', 'busy')"
            )['count'],
            'tasks_created_today' => (int) $this->db->fetchOne(
                "SELECT COUNT(*) as count FROM tasks WHERE DATE(created_at) = ?",
                [$today]
            )['count'],
            'tasks_completed_today' => (int) $this->db->fetchOne(
                "SELECT COUNT(*) as count FROM tasks WHERE DATE(end_time) = ? AND status = 'completed'",
                [$today]
            )['count']
        ];
    }
}
