<?php
/**
 * Notification Controller
 */

class NotificationController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $id, ?string $action, array $input): void {
        $user = AuthMiddleware::authenticate();

        if ($id === 'read-all' && $method === 'POST') {
            $this->markAllAsRead();
            return;
        }

        if ($id === 'unread-count' && $method === 'GET') {
            $this->getUnreadCount();
            return;
        }

        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getNotification((int) $id);
                } else {
                    $this->listNotifications();
                }
                break;
            case 'PUT':
                $this->markAsRead((int) $id);
                break;
            case 'DELETE':
                $this->deleteNotification((int) $id);
                break;
            default:
                Response::error('Method not allowed', 405);
        }
    }

    private function listNotifications(): void {
        $userId = AuthMiddleware::getUserId();
        $page = (int) ($_GET['page'] ?? 1);
        $perPage = min((int) ($_GET['per_page'] ?? DEFAULT_PAGE_SIZE), MAX_PAGE_SIZE);

        $result = NotificationService::getUserNotifications($userId, $page, $perPage);

        Response::paginated(
            $result['notifications'],
            $result['total'],
            $page,
            $perPage
        );
    }

    private function getNotification(int $id): void {
        $userId = AuthMiddleware::getUserId();

        $notification = $this->db->fetchOne(
            "SELECT * FROM notifications WHERE id = ? AND user_id = ?",
            [$id, $userId]
        );

        if (!$notification) {
            Response::notFound('Notification not found');
        }

        // Parse JSON data
        if ($notification['data']) {
            $notification['data'] = json_decode($notification['data'], true);
        }

        Response::success($notification);
    }

    private function markAsRead(int $id): void {
        $userId = AuthMiddleware::getUserId();
        
        $success = NotificationService::markAsRead($id, $userId);

        if (!$success) {
            Response::notFound('Notification not found');
        }

        Response::success(null, 'Notification marked as read');
    }

    private function markAllAsRead(): void {
        $userId = AuthMiddleware::getUserId();
        
        $count = NotificationService::markAllAsRead($userId);

        Response::success(['marked_count' => $count], 'All notifications marked as read');
    }

    private function getUnreadCount(): void {
        $userId = AuthMiddleware::getUserId();
        $count = NotificationService::getUnreadCount($userId);

        Response::success(['unread_count' => $count]);
    }

    private function deleteNotification(int $id): void {
        $userId = AuthMiddleware::getUserId();

        $deleted = $this->db->delete('notifications', 'id = ? AND user_id = ?', [$id, $userId]);

        if (!$deleted) {
            Response::notFound('Notification not found');
        }

        Response::success(null, 'Notification deleted');
    }
}
