<?php
/**
 * Recurring Task Controller
 */

class RecurringTaskController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $id, ?string $action, array $input): void {
        AuthMiddleware::requireManager();

        if ($action === 'run' && $method === 'POST') {
            $this->runRecurringTasks();
            return;
        }

        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getRecurringTask((int) $id);
                } else {
                    $this->listRecurringTasks();
                }
                break;
            case 'POST':
                $this->createRecurringTask($input);
                break;
            case 'PUT':
                $this->updateRecurringTask((int) $id, $input);
                break;
            case 'DELETE':
                $this->deleteRecurringTask((int) $id);
                break;
            default:
                Response::error('Method not allowed', 405);
        }
    }

    private function listRecurringTasks(): void {
        $tasks = $this->db->fetchAll(
            "SELECT rt.*, u.name as created_by_name 
             FROM recurring_tasks rt 
             JOIN users u ON rt.created_by = u.id 
             ORDER BY rt.created_at DESC"
        );

        foreach ($tasks as &$task) {
            if ($task['assign_to']) {
                $task['assign_to'] = json_decode($task['assign_to'], true);
            }
        }

        Response::success($tasks);
    }

    private function getRecurringTask(int $id): void {
        $task = $this->db->fetchOne(
            "SELECT rt.*, u.name as created_by_name 
             FROM recurring_tasks rt 
             JOIN users u ON rt.created_by = u.id 
             WHERE rt.id = ?",
            [$id]
        );

        if (!$task) {
            Response::notFound('Recurring task not found');
        }

        if ($task['assign_to']) {
            $task['assign_to'] = json_decode($task['assign_to'], true);
        }

        // Get generated tasks
        $task['generated_tasks'] = $this->db->fetchAll(
            "SELECT task_id, title, status, created_at 
             FROM tasks 
             WHERE recurring_task_id = ? 
             ORDER BY created_at DESC 
             LIMIT 10",
            [$id]
        );

        Response::success($task);
    }

    private function createRecurringTask(array $input): void {
        $validator = new Validator($input);
        $validator
            ->required('title')
            ->required('frequency')
            ->in('frequency', ['daily', 'weekly', 'monthly'])
            ->in('type', ['support', 'maintenance', 'visit', 'internal', 'urgent'])
            ->in('priority', ['low', 'medium', 'high', 'urgent'])
            ->validate();

        $userId = AuthMiddleware::getUserId();

        // Calculate next run date
        $nextRun = $this->calculateNextRun(
            $input['frequency'],
            $input['day_of_week'] ?? null,
            $input['day_of_month'] ?? null
        );

        $id = $this->db->insert('recurring_tasks', [
            'title' => $input['title'],
            'description' => $input['description'] ?? null,
            'type' => $input['type'] ?? 'maintenance',
            'priority' => $input['priority'] ?? 'medium',
            'frequency' => $input['frequency'],
            'day_of_week' => $input['day_of_week'] ?? null,
            'day_of_month' => $input['day_of_month'] ?? null,
            'time_of_day' => $input['time_of_day'] ?? null,
            'assign_to' => !empty($input['assign_to']) ? json_encode($input['assign_to']) : null,
            'created_by' => $userId,
            'is_active' => 1,
            'next_run' => $nextRun
        ]);

        AuditService::log('create', 'recurring_task', $id, null, $input);

        $task = $this->db->fetchOne("SELECT * FROM recurring_tasks WHERE id = ?", [$id]);

        Response::created($task);
    }

    private function updateRecurringTask(int $id, array $input): void {
        $task = $this->db->fetchOne("SELECT * FROM recurring_tasks WHERE id = ?", [$id]);

        if (!$task) {
            Response::notFound('Recurring task not found');
        }

        $updateData = [];
        $allowedFields = ['title', 'description', 'type', 'priority', 'frequency', 'day_of_week', 'day_of_month', 'time_of_day', 'is_active'];

        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateData[$field] = $input[$field];
            }
        }

        if (isset($input['assign_to'])) {
            $updateData['assign_to'] = json_encode($input['assign_to']);
        }

        // Recalculate next run if frequency changed
        if (isset($updateData['frequency']) || isset($updateData['day_of_week']) || isset($updateData['day_of_month'])) {
            $updateData['next_run'] = $this->calculateNextRun(
                $updateData['frequency'] ?? $task['frequency'],
                $updateData['day_of_week'] ?? $task['day_of_week'],
                $updateData['day_of_month'] ?? $task['day_of_month']
            );
        }

        if (empty($updateData)) {
            Response::error('No valid fields to update');
        }

        $this->db->update('recurring_tasks', $updateData, 'id = ?', [$id]);

        AuditService::log('update', 'recurring_task', $id, $task, $updateData);

        $updated = $this->db->fetchOne("SELECT * FROM recurring_tasks WHERE id = ?", [$id]);
        Response::success($updated, 'Recurring task updated');
    }

    private function deleteRecurringTask(int $id): void {
        $task = $this->db->fetchOne("SELECT * FROM recurring_tasks WHERE id = ?", [$id]);

        if (!$task) {
            Response::notFound('Recurring task not found');
        }

        $this->db->delete('recurring_tasks', 'id = ?', [$id]);

        AuditService::log('delete', 'recurring_task', $id, $task);

        Response::success(null, 'Recurring task deleted');
    }

    private function runRecurringTasks(): void {
        $today = date('Y-m-d');
        
        $tasks = $this->db->fetchAll(
            "SELECT * FROM recurring_tasks WHERE is_active = 1 AND (next_run IS NULL OR next_run <= ?)",
            [$today]
        );

        $created = 0;

        foreach ($tasks as $recurringTask) {
            // Check if already generated for today
            $existing = $this->db->fetchOne(
                "SELECT id FROM tasks WHERE recurring_task_id = ? AND DATE(created_at) = ?",
                [$recurringTask['id'], $today]
            );

            if ($existing) {
                continue;
            }

            // Generate task ID
            $this->db->query("CALL generate_task_id(@new_id)");
            $result = $this->db->fetchOne("SELECT @new_id as task_id");
            $taskId = $result['task_id'];

            // Create task
            $newTaskId = $this->db->insert('tasks', [
                'task_id' => $taskId,
                'title' => $recurringTask['title'],
                'description' => $recurringTask['description'],
                'type' => $recurringTask['type'],
                'priority' => $recurringTask['priority'],
                'status' => 'new',
                'created_by' => $recurringTask['created_by'],
                'is_recurring' => 1,
                'recurring_task_id' => $recurringTask['id'],
                'sla_deadline' => date('Y-m-d H:i:s', strtotime('+' . DEFAULT_SLA_HOURS . ' hours'))
            ]);

            // Assign users
            if ($recurringTask['assign_to']) {
                $assignees = json_decode($recurringTask['assign_to'], true);
                foreach ($assignees as $index => $assigneeId) {
                    $this->db->insert('task_assignments', [
                        'task_id' => $newTaskId,
                        'user_id' => $assigneeId,
                        'assigned_by' => $recurringTask['created_by'],
                        'is_primary' => $index === 0 ? 1 : 0
                    ]);

                    // Notify
                    $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$newTaskId]);
                    NotificationService::taskAssigned($task, $assigneeId);
                }
            }

            // Update next run
            $nextRun = $this->calculateNextRun(
                $recurringTask['frequency'],
                $recurringTask['day_of_week'],
                $recurringTask['day_of_month']
            );

            $this->db->update('recurring_tasks', [
                'last_generated' => $today,
                'next_run' => $nextRun
            ], 'id = ?', [$recurringTask['id']]);

            $created++;
        }

        Response::success(['created_count' => $created], "Generated {$created} tasks");
    }

    private function calculateNextRun(string $frequency, ?int $dayOfWeek, ?int $dayOfMonth): string {
        $today = new DateTime();
        
        switch ($frequency) {
            case 'daily':
                $today->modify('+1 day');
                break;
            case 'weekly':
                if ($dayOfWeek !== null) {
                    $days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                    $today->modify('next ' . $days[$dayOfWeek]);
                } else {
                    $today->modify('+1 week');
                }
                break;
            case 'monthly':
                if ($dayOfMonth !== null) {
                    $today->modify('first day of next month');
                    $today->setDate($today->format('Y'), $today->format('m'), min($dayOfMonth, $today->format('t')));
                } else {
                    $today->modify('+1 month');
                }
                break;
        }

        return $today->format('Y-m-d');
    }
}
