<?php
/**
 * Task Controller
 */

class TaskController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $id, ?string $action, array $input): void {
        $user = AuthMiddleware::authenticate();

        // Handle special actions
        if ($id && $action) {
            switch ($action) {
                case 'start':
                    if ($method === 'POST') $this->startTask((int) $id, $input);
                    return;
                case 'complete':
                    if ($method === 'POST') $this->completeTask((int) $id, $input);
                    return;
                case 'hold':
                    if ($method === 'POST') $this->holdTask((int) $id, $input);
                    return;
                case 'resume':
                    if ($method === 'POST') $this->resumeTask((int) $id, $input);
                    return;
                case 'cancel':
                    if ($method === 'POST') $this->cancelTask((int) $id, $input);
                    return;
                case 'assign':
                    if ($method === 'POST') $this->assignTask((int) $id, $input);
                    return;
                case 'logs':
                    if ($method === 'GET') $this->getTaskLogs((int) $id);
                    return;
                case 'attachments':
                    if ($method === 'GET') $this->getTaskAttachments((int) $id);
                    return;
                case 'voice-note':
                    if ($method === 'POST') $this->addVoiceNote((int) $id, $input);
                    return;
            }
        }

        // Handle CRUD operations
        switch ($method) {
            case 'GET':
                if ($id === 'my') {
                    $this->getMyTasks();
                } elseif ($id) {
                    $this->getTask((int) $id);
                } else {
                    $this->listTasks();
                }
                break;
            case 'POST':
                $this->createTask($input);
                break;
            case 'PUT':
                $this->updateTask((int) $id, $input);
                break;
            case 'DELETE':
                AuthMiddleware::requireManager();
                $this->deleteTask((int) $id);
                break;
            default:
                Response::error('Method not allowed', 405);
        }
    }

    private function generateTaskId(): string {
        $this->db->query("CALL generate_task_id(@new_id)");
        $result = $this->db->fetchOne("SELECT @new_id as task_id");
        return $result['task_id'];
    }

    private function listTasks(): void {
        $page = (int) ($_GET['page'] ?? 1);
        $perPage = min((int) ($_GET['per_page'] ?? DEFAULT_PAGE_SIZE), MAX_PAGE_SIZE);
        $offset = ($page - 1) * $perPage;

        $where = ['1=1'];
        $params = [];

        // Filter by status
        if (!empty($_GET['status'])) {
            $where[] = 't.status = ?';
            $params[] = $_GET['status'];
        }

        // Filter by type
        if (!empty($_GET['type'])) {
            $where[] = 't.type = ?';
            $params[] = $_GET['type'];
        }

        // Filter by priority
        if (!empty($_GET['priority'])) {
            $where[] = 't.priority = ?';
            $params[] = $_GET['priority'];
        }

        // Filter by assigned user
        if (!empty($_GET['assigned_to'])) {
            $where[] = 'EXISTS (SELECT 1 FROM task_assignments ta WHERE ta.task_id = t.id AND ta.user_id = ?)';
            $params[] = $_GET['assigned_to'];
        }

        // Filter by date range
        if (!empty($_GET['date_from'])) {
            $where[] = 't.created_at >= ?';
            $params[] = $_GET['date_from'] . ' 00:00:00';
        }
        if (!empty($_GET['date_to'])) {
            $where[] = 't.created_at <= ?';
            $params[] = $_GET['date_to'] . ' 23:59:59';
        }

        // Search
        if (!empty($_GET['search'])) {
            $where[] = '(t.title LIKE ? OR t.task_id LIKE ? OR t.description LIKE ?)';
            $search = '%' . $_GET['search'] . '%';
            $params = array_merge($params, [$search, $search, $search]);
        }

        $whereClause = implode(' AND ', $where);

        $tasks = $this->db->fetchAll(
            "SELECT t.*, u.name as created_by_name,
                    (SELECT GROUP_CONCAT(u2.name SEPARATOR ', ') 
                     FROM task_assignments ta 
                     JOIN users u2 ON ta.user_id = u2.id 
                     WHERE ta.task_id = t.id) as assigned_to_names
             FROM tasks t
             JOIN users u ON t.created_by = u.id
             WHERE {$whereClause}
             ORDER BY 
                CASE t.priority 
                    WHEN 'urgent' THEN 1 
                    WHEN 'high' THEN 2 
                    WHEN 'medium' THEN 3 
                    ELSE 4 
                END,
                t.created_at DESC
             LIMIT ? OFFSET ?",
            array_merge($params, [$perPage, $offset])
        );

        $total = $this->db->fetchOne(
            "SELECT COUNT(*) as count FROM tasks t WHERE {$whereClause}",
            $params
        );

        Response::paginated($tasks, (int) $total['count'], $page, $perPage);
    }

    private function getMyTasks(): void {
        $userId = AuthMiddleware::getUserId();
        $status = $_GET['status'] ?? null;

        $where = ['ta.user_id = ?'];
        $params = [$userId];

        if ($status) {
            $where[] = 't.status = ?';
            $params[] = $status;
        }

        $whereClause = implode(' AND ', $where);

        $tasks = $this->db->fetchAll(
            "SELECT t.*, u.name as created_by_name
             FROM tasks t
             JOIN task_assignments ta ON t.id = ta.task_id
             JOIN users u ON t.created_by = u.id
             WHERE {$whereClause}
             ORDER BY 
                CASE t.priority 
                    WHEN 'urgent' THEN 1 
                    WHEN 'high' THEN 2 
                    WHEN 'medium' THEN 3 
                    ELSE 4 
                END,
                t.created_at DESC",
            $params
        );

        Response::success($tasks);
    }

    private function getTask(int $id): void {
        $task = $this->db->fetchOne(
            "SELECT t.*, u.name as created_by_name
             FROM tasks t
             JOIN users u ON t.created_by = u.id
             WHERE t.id = ?",
            [$id]
        );

        if (!$task) {
            Response::notFound('Task not found');
        }

        // Get assignments
        $task['assignments'] = $this->db->fetchAll(
            "SELECT ta.*, u.name, u.email 
             FROM task_assignments ta 
             JOIN users u ON ta.user_id = u.id 
             WHERE ta.task_id = ?",
            [$id]
        );

        // Get attachments count
        $attachments = $this->db->fetchOne(
            "SELECT COUNT(*) as count FROM task_attachments WHERE task_id = ?",
            [$id]
        );
        $task['attachments_count'] = (int) $attachments['count'];

        // Get voice notes count
        $voiceNotes = $this->db->fetchOne(
            "SELECT COUNT(*) as count FROM voice_notes WHERE task_id = ?",
            [$id]
        );
        $task['voice_notes_count'] = (int) $voiceNotes['count'];

        // Get recent logs
        $task['recent_logs'] = $this->db->fetchAll(
            "SELECT tl.*, u.name as user_name 
             FROM task_logs tl 
             JOIN users u ON tl.user_id = u.id 
             WHERE tl.task_id = ? 
             ORDER BY tl.created_at DESC 
             LIMIT 5",
            [$id]
        );

        Response::success($task);
    }

    private function createTask(array $input): void {
        $validator = new Validator($input);
        $validator
            ->required('title')
            ->maxLength('title', 255)
            ->required('type')
            ->in('type', ['support', 'maintenance', 'visit', 'internal', 'urgent'])
            ->in('priority', ['low', 'medium', 'high', 'urgent'])
            ->validate();

        $userId = AuthMiddleware::getUserId();
        $taskId = $this->generateTaskId();

        // Calculate SLA deadline
        $slaHours = $input['sla_hours'] ?? DEFAULT_SLA_HOURS;
        $slaDeadline = date('Y-m-d H:i:s', strtotime("+{$slaHours} hours"));

        $this->db->beginTransaction();

        try {
            $id = $this->db->insert('tasks', [
                'task_id' => $taskId,
                'title' => $input['title'],
                'description' => $input['description'] ?? null,
                'type' => $input['type'],
                'priority' => $input['priority'] ?? 'medium',
                'status' => 'new',
                'created_by' => $userId,
                'location_name' => $input['location_name'] ?? null,
                'gps_lat_start' => $input['gps_lat'] ?? null,
                'gps_lng_start' => $input['gps_lng'] ?? null,
                'sla_deadline' => $slaDeadline
            ]);

            // Assign users if provided
            $assignees = $input['assign_to'] ?? [];
            if (!empty($assignees)) {
                foreach ($assignees as $index => $assigneeId) {
                    $this->db->insert('task_assignments', [
                        'task_id' => $id,
                        'user_id' => $assigneeId,
                        'assigned_by' => $userId,
                        'is_primary' => $index === 0 ? 1 : 0
                    ]);
                }
            }

            // Log creation
            AuditService::taskLog($id, 'created', null, 'new');

            $this->db->commit();

            $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);

            // Send notifications
            NotificationService::taskCreated($task, $userId);
            foreach ($assignees as $assigneeId) {
                NotificationService::taskAssigned($task, $assigneeId);
            }

            Response::created($task);
        } catch (Exception $e) {
            $this->db->rollback();
            throw $e;
        }
    }

    private function updateTask(int $id, array $input): void {
        $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        
        if (!$task) {
            Response::notFound('Task not found');
        }

        $updateData = [];
        $allowedFields = ['title', 'description', 'type', 'priority', 'location_name'];

        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateData[$field] = $input[$field];
            }
        }

        if (empty($updateData)) {
            Response::error('No valid fields to update');
        }

        $this->db->update('tasks', $updateData, 'id = ?', [$id]);

        AuditService::taskLog($id, 'updated', json_encode($task), json_encode($updateData));

        $updatedTask = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);

        Response::success($updatedTask, 'Task updated');
    }

    private function startTask(int $id, array $input): void {
        $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        
        if (!$task) {
            Response::notFound('Task not found');
        }

        if (!in_array($task['status'], ['new', 'on_hold'])) {
            Response::error('Task cannot be started from current status');
        }

        $userId = AuthMiddleware::getUserId();
        $now = date('Y-m-d H:i:s');

        $this->db->update('tasks', [
            'status' => 'started',
            'start_time' => $task['start_time'] ?? $now,
            'gps_lat_start' => $input['gps_lat'] ?? $task['gps_lat_start'],
            'gps_lng_start' => $input['gps_lng'] ?? $task['gps_lng_start']
        ], 'id = ?', [$id]);

        // Log time tracking
        $this->db->insert('time_tracking', [
            'task_id' => $id,
            'user_id' => $userId,
            'action' => $task['status'] === 'on_hold' ? 'resume' : 'start',
            'timestamp' => $now,
            'gps_lat' => $input['gps_lat'] ?? null,
            'gps_lng' => $input['gps_lng'] ?? null
        ]);

        // Update user status
        $this->db->query(
            "UPDATE user_status SET status = 'on_task', current_task_id = ? WHERE user_id = ?",
            [$id, $userId]
        );

        AuditService::taskLog(
            $id, 
            'started', 
            $task['status'], 
            'started', 
            null, 
            $input['gps_lat'] ?? null, 
            $input['gps_lng'] ?? null
        );

        $updatedTask = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        NotificationService::taskStarted($updatedTask);

        Response::success($updatedTask, 'Task started');
    }

    private function completeTask(int $id, array $input): void {
        $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        
        if (!$task) {
            Response::notFound('Task not found');
        }

        if ($task['status'] !== 'started') {
            Response::error('Only started tasks can be completed');
        }

        $userId = AuthMiddleware::getUserId();
        $now = date('Y-m-d H:i:s');

        // Calculate duration
        $startTime = new DateTime($task['start_time']);
        $endTime = new DateTime($now);
        $duration = $startTime->diff($endTime);
        $durationMinutes = ($duration->days * 24 * 60) + ($duration->h * 60) + $duration->i;

        // Check SLA breach
        $slaBreached = $task['sla_deadline'] && $now > $task['sla_deadline'] ? 1 : 0;

        $this->db->update('tasks', [
            'status' => 'completed',
            'end_time' => $now,
            'duration_minutes' => $durationMinutes,
            'gps_lat_end' => $input['gps_lat'] ?? null,
            'gps_lng_end' => $input['gps_lng'] ?? null,
            'sla_breached' => $slaBreached
        ], 'id = ?', [$id]);

        // Log time tracking
        $this->db->insert('time_tracking', [
            'task_id' => $id,
            'user_id' => $userId,
            'action' => 'stop',
            'timestamp' => $now,
            'gps_lat' => $input['gps_lat'] ?? null,
            'gps_lng' => $input['gps_lng'] ?? null,
            'notes' => $input['notes'] ?? null
        ]);

        // Update user status
        $this->db->query(
            "UPDATE user_status SET status = 'online', current_task_id = NULL WHERE user_id = ?",
            [$userId]
        );

        AuditService::taskLog(
            $id, 
            'completed', 
            'started', 
            'completed', 
            $input['notes'] ?? null, 
            $input['gps_lat'] ?? null, 
            $input['gps_lng'] ?? null
        );

        $updatedTask = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        NotificationService::taskCompleted($updatedTask);

        Response::success($updatedTask, 'Task completed');
    }

    private function holdTask(int $id, array $input): void {
        $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        
        if (!$task) {
            Response::notFound('Task not found');
        }

        if ($task['status'] !== 'started') {
            Response::error('Only started tasks can be put on hold');
        }

        $validator = new Validator($input);
        $validator->required('hold_reason')->validate();

        $userId = AuthMiddleware::getUserId();

        $this->db->update('tasks', [
            'status' => 'on_hold',
            'hold_reason' => $input['hold_reason']
        ], 'id = ?', [$id]);

        // Log time tracking
        $this->db->insert('time_tracking', [
            'task_id' => $id,
            'user_id' => $userId,
            'action' => 'pause',
            'timestamp' => date('Y-m-d H:i:s'),
            'notes' => $input['hold_reason']
        ]);

        // Update user status
        $this->db->query(
            "UPDATE user_status SET status = 'online', current_task_id = NULL WHERE user_id = ?",
            [$userId]
        );

        AuditService::taskLog($id, 'on_hold', 'started', 'on_hold', $input['hold_reason']);

        $updatedTask = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);

        Response::success($updatedTask, 'Task put on hold');
    }

    private function resumeTask(int $id, array $input): void {
        $this->startTask($id, $input);
    }

    private function cancelTask(int $id, array $input): void {
        AuthMiddleware::requireManager();

        $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        
        if (!$task) {
            Response::notFound('Task not found');
        }

        if ($task['status'] === 'completed') {
            Response::error('Completed tasks cannot be cancelled');
        }

        $this->db->update('tasks', ['status' => 'cancelled'], 'id = ?', [$id]);

        AuditService::taskLog($id, 'cancelled', $task['status'], 'cancelled', $input['reason'] ?? null);

        Response::success(null, 'Task cancelled');
    }

    private function assignTask(int $id, array $input): void {
        AuthMiddleware::requireManager();

        $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        
        if (!$task) {
            Response::notFound('Task not found');
        }

        $validator = new Validator($input);
        $validator->required('user_ids')->array('user_ids')->validate();

        $userId = AuthMiddleware::getUserId();

        // Remove existing assignments
        $this->db->delete('task_assignments', 'task_id = ?', [$id]);

        // Add new assignments
        foreach ($input['user_ids'] as $index => $assigneeId) {
            $this->db->insert('task_assignments', [
                'task_id' => $id,
                'user_id' => $assigneeId,
                'assigned_by' => $userId,
                'is_primary' => $index === 0 ? 1 : 0
            ]);

            NotificationService::taskAssigned($task, $assigneeId);
        }

        AuditService::taskLog($id, 'assigned', null, json_encode($input['user_ids']));

        Response::success(null, 'Task assigned');
    }

    private function getTaskLogs(int $id): void {
        $logs = AuditService::getTaskLogs($id);
        Response::success($logs);
    }

    private function getTaskAttachments(int $id): void {
        $attachments = $this->db->fetchAll(
            "SELECT ta.*, u.name as uploaded_by_name 
             FROM task_attachments ta 
             JOIN users u ON ta.user_id = u.id 
             WHERE ta.task_id = ? 
             ORDER BY ta.created_at DESC",
            [$id]
        );

        foreach ($attachments as &$attachment) {
            $attachment['url'] = FileService::getUrl($attachment['file_path']);
        }

        Response::success($attachments);
    }

    private function addVoiceNote(int $id, array $input): void {
        $task = $this->db->fetchOne("SELECT id FROM tasks WHERE id = ?", [$id]);
        
        if (!$task) {
            Response::notFound('Task not found');
        }

        $validator = new Validator($input);
        $validator->required('audio_data')->validate();

        $userId = AuthMiddleware::getUserId();

        // Upload voice note
        $fileInfo = FileService::uploadBase64($input['audio_data'], 'voice_notes', 'm4a');

        $voiceNoteId = $this->db->insert('voice_notes', [
            'task_id' => $id,
            'user_id' => $userId,
            'file_path' => $fileInfo['file_path'],
            'duration_seconds' => $input['duration_seconds'] ?? null
        ]);

        // Also add to attachments
        $this->db->insert('task_attachments', [
            'task_id' => $id,
            'user_id' => $userId,
            'file_type' => 'voice',
            'file_name' => $fileInfo['file_name'],
            'file_path' => $fileInfo['file_path'],
            'file_size' => $fileInfo['file_size'],
            'mime_type' => $fileInfo['mime_type']
        ]);

        AuditService::taskLog($id, 'voice_note_added');

        Response::created([
            'id' => $voiceNoteId,
            'url' => FileService::getUrl($fileInfo['file_path'])
        ], 'Voice note added');
    }

    private function deleteTask(int $id): void {
        $task = $this->db->fetchOne("SELECT * FROM tasks WHERE id = ?", [$id]);
        
        if (!$task) {
            Response::notFound('Task not found');
        }

        $this->db->delete('tasks', 'id = ?', [$id]);

        AuditService::log('delete', 'task', $id, $task);

        Response::success(null, 'Task deleted');
    }
}
