<?php
/**
 * Task Template Controller
 */

class TemplateController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $id, array $input): void {
        $user = AuthMiddleware::authenticate();

        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getTemplate((int) $id);
                } else {
                    $this->listTemplates();
                }
                break;
            case 'POST':
                AuthMiddleware::requireManager();
                $this->createTemplate($input);
                break;
            case 'PUT':
                AuthMiddleware::requireManager();
                $this->updateTemplate((int) $id, $input);
                break;
            case 'DELETE':
                AuthMiddleware::requireManager();
                $this->deleteTemplate((int) $id);
                break;
            default:
                Response::error('Method not allowed', 405);
        }
    }

    private function listTemplates(): void {
        $templates = $this->db->fetchAll(
            "SELECT tt.*, u.name as created_by_name 
             FROM task_templates tt 
             JOIN users u ON tt.created_by = u.id 
             WHERE tt.is_active = 1 
             ORDER BY tt.name"
        );

        foreach ($templates as &$template) {
            if ($template['default_assignees']) {
                $template['default_assignees'] = json_decode($template['default_assignees'], true);
            }
        }

        Response::success($templates);
    }

    private function getTemplate(int $id): void {
        $template = $this->db->fetchOne(
            "SELECT tt.*, u.name as created_by_name 
             FROM task_templates tt 
             JOIN users u ON tt.created_by = u.id 
             WHERE tt.id = ?",
            [$id]
        );

        if (!$template) {
            Response::notFound('Template not found');
        }

        if ($template['default_assignees']) {
            $template['default_assignees'] = json_decode($template['default_assignees'], true);
        }

        Response::success($template);
    }

    private function createTemplate(array $input): void {
        $validator = new Validator($input);
        $validator
            ->required('name')
            ->required('title')
            ->required('type')
            ->in('type', ['support', 'maintenance', 'visit', 'internal', 'urgent'])
            ->in('priority', ['low', 'medium', 'high', 'urgent'])
            ->validate();

        $userId = AuthMiddleware::getUserId();

        $id = $this->db->insert('task_templates', [
            'name' => $input['name'],
            'title' => $input['title'],
            'description' => $input['description'] ?? null,
            'type' => $input['type'],
            'priority' => $input['priority'] ?? 'medium',
            'default_assignees' => !empty($input['default_assignees']) ? json_encode($input['default_assignees']) : null,
            'sla_hours' => $input['sla_hours'] ?? null,
            'created_by' => $userId,
            'is_active' => 1
        ]);

        AuditService::log('create', 'task_template', $id, null, $input);

        $template = $this->db->fetchOne("SELECT * FROM task_templates WHERE id = ?", [$id]);

        Response::created($template);
    }

    private function updateTemplate(int $id, array $input): void {
        $template = $this->db->fetchOne("SELECT * FROM task_templates WHERE id = ?", [$id]);

        if (!$template) {
            Response::notFound('Template not found');
        }

        $updateData = [];
        $allowedFields = ['name', 'title', 'description', 'type', 'priority', 'sla_hours', 'is_active'];

        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateData[$field] = $input[$field];
            }
        }

        if (isset($input['default_assignees'])) {
            $updateData['default_assignees'] = json_encode($input['default_assignees']);
        }

        if (empty($updateData)) {
            Response::error('No valid fields to update');
        }

        $this->db->update('task_templates', $updateData, 'id = ?', [$id]);

        AuditService::log('update', 'task_template', $id, $template, $updateData);

        $updated = $this->db->fetchOne("SELECT * FROM task_templates WHERE id = ?", [$id]);
        Response::success($updated, 'Template updated');
    }

    private function deleteTemplate(int $id): void {
        $template = $this->db->fetchOne("SELECT * FROM task_templates WHERE id = ?", [$id]);

        if (!$template) {
            Response::notFound('Template not found');
        }

        // Soft delete
        $this->db->update('task_templates', ['is_active' => 0], 'id = ?', [$id]);

        AuditService::log('delete', 'task_template', $id, $template);

        Response::success(null, 'Template deleted');
    }
}
