<?php
/**
 * Upload Controller
 */

class UploadController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $type, array $input): void {
        $user = AuthMiddleware::authenticate();

        if ($method !== 'POST') {
            Response::error('Method not allowed', 405);
        }

        switch ($type) {
            case 'task-attachment':
                $this->uploadTaskAttachment($input);
                break;
            case 'voice-note':
                $this->uploadVoiceNote($input);
                break;
            case 'avatar':
                $this->uploadAvatar($input);
                break;
            case 'evidence':
                $this->uploadEvidence($input);
                break;
            default:
                $this->uploadGeneral();
        }
    }

    private function uploadTaskAttachment(array $input): void {
        $validator = new Validator($input);
        $validator->required('task_id')->numeric('task_id')->validate();

        $task = $this->db->fetchOne("SELECT id FROM tasks WHERE id = ?", [$input['task_id']]);
        if (!$task) {
            Response::notFound('Task not found');
        }

        if (empty($_FILES['file'])) {
            Response::error('No file uploaded');
        }

        $userId = AuthMiddleware::getUserId();
        $fileInfo = FileService::upload($_FILES['file'], 'task_attachments');

        $attachmentId = $this->db->insert('task_attachments', [
            'task_id' => $input['task_id'],
            'user_id' => $userId,
            'file_type' => $fileInfo['file_type'],
            'file_name' => $fileInfo['file_name'],
            'file_path' => $fileInfo['file_path'],
            'file_size' => $fileInfo['file_size'],
            'mime_type' => $fileInfo['mime_type'],
            'is_evidence' => $input['is_evidence'] ?? 0
        ]);

        AuditService::taskLog($input['task_id'], 'attachment_added', null, $fileInfo['file_name']);

        Response::created([
            'id' => $attachmentId,
            'url' => FileService::getUrl($fileInfo['file_path']),
            'file_name' => $fileInfo['file_name'],
            'file_type' => $fileInfo['file_type']
        ]);
    }

    private function uploadVoiceNote(array $input): void {
        $validator = new Validator($input);
        $validator->required('task_id')->numeric('task_id')->validate();

        $task = $this->db->fetchOne("SELECT id FROM tasks WHERE id = ?", [$input['task_id']]);
        if (!$task) {
            Response::notFound('Task not found');
        }

        $userId = AuthMiddleware::getUserId();

        // Handle base64 audio data
        if (!empty($input['audio_data'])) {
            $fileInfo = FileService::uploadBase64($input['audio_data'], 'voice_notes', 'm4a');
        } elseif (!empty($_FILES['file'])) {
            $fileInfo = FileService::upload($_FILES['file'], 'voice_notes');
        } else {
            Response::error('No audio data provided');
        }

        $voiceNoteId = $this->db->insert('voice_notes', [
            'task_id' => $input['task_id'],
            'user_id' => $userId,
            'file_path' => $fileInfo['file_path'],
            'duration_seconds' => $input['duration_seconds'] ?? null
        ]);

        // Also add to attachments
        $this->db->insert('task_attachments', [
            'task_id' => $input['task_id'],
            'user_id' => $userId,
            'file_type' => 'voice',
            'file_name' => $fileInfo['file_name'],
            'file_path' => $fileInfo['file_path'],
            'file_size' => $fileInfo['file_size'],
            'mime_type' => $fileInfo['mime_type']
        ]);

        AuditService::taskLog($input['task_id'], 'voice_note_added');

        Response::created([
            'id' => $voiceNoteId,
            'url' => FileService::getUrl($fileInfo['file_path'])
        ]);
    }

    private function uploadAvatar(array $input): void {
        if (empty($_FILES['file'])) {
            Response::error('No file uploaded');
        }

        $userId = AuthMiddleware::getUserId();
        $fileInfo = FileService::upload($_FILES['file'], 'avatars');

        // Delete old avatar if exists
        $user = $this->db->fetchOne("SELECT avatar FROM users WHERE id = ?", [$userId]);
        if ($user['avatar']) {
            FileService::delete($user['avatar']);
        }

        $this->db->update('users', ['avatar' => $fileInfo['file_path']], 'id = ?', [$userId]);

        Response::success([
            'avatar_url' => FileService::getUrl($fileInfo['file_path'])
        ], 'Avatar updated');
    }

    private function uploadEvidence(array $input): void {
        $validator = new Validator($input);
        $validator->required('task_id')->numeric('task_id')->validate();

        $task = $this->db->fetchOne("SELECT id FROM tasks WHERE id = ?", [$input['task_id']]);
        if (!$task) {
            Response::notFound('Task not found');
        }

        if (empty($_FILES['file'])) {
            Response::error('No file uploaded');
        }

        $userId = AuthMiddleware::getUserId();
        $fileInfo = FileService::upload($_FILES['file'], 'evidence');

        $attachmentId = $this->db->insert('task_attachments', [
            'task_id' => $input['task_id'],
            'user_id' => $userId,
            'file_type' => $fileInfo['file_type'],
            'file_name' => $fileInfo['file_name'],
            'file_path' => $fileInfo['file_path'],
            'file_size' => $fileInfo['file_size'],
            'mime_type' => $fileInfo['mime_type'],
            'is_evidence' => 1
        ]);

        AuditService::taskLog($input['task_id'], 'evidence_added', null, $fileInfo['file_name']);

        Response::created([
            'id' => $attachmentId,
            'url' => FileService::getUrl($fileInfo['file_path']),
            'file_name' => $fileInfo['file_name']
        ]);
    }

    private function uploadGeneral(): void {
        if (empty($_FILES['file'])) {
            Response::error('No file uploaded');
        }

        $fileInfo = FileService::upload($_FILES['file'], 'general');

        Response::created([
            'url' => FileService::getUrl($fileInfo['file_path']),
            'file_name' => $fileInfo['file_name'],
            'file_type' => $fileInfo['file_type'],
            'file_size' => $fileInfo['file_size']
        ]);
    }
}
