<?php
/**
 * User Controller
 */

class UserController {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function handle(string $method, ?string $id, ?string $action, array $input): void {
        $user = AuthMiddleware::authenticate();

        if ($action === 'status' && $method === 'PUT') {
            $this->updateStatus($input);
            return;
        }

        if ($id === 'staff' && $method === 'GET') {
            $this->getStaffList();
            return;
        }

        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getUser((int) $id);
                } else {
                    $this->listUsers();
                }
                break;
            case 'POST':
                AuthMiddleware::requireAdmin();
                $this->createUser($input);
                break;
            case 'PUT':
                $this->updateUser((int) $id, $input);
                break;
            case 'DELETE':
                AuthMiddleware::requireAdmin();
                $this->deleteUser((int) $id);
                break;
            default:
                Response::error('Method not allowed', 405);
        }
    }

    private function listUsers(): void {
        AuthMiddleware::requireManager();

        $page = (int) ($_GET['page'] ?? 1);
        $perPage = min((int) ($_GET['per_page'] ?? DEFAULT_PAGE_SIZE), MAX_PAGE_SIZE);
        $offset = ($page - 1) * $perPage;

        $where = [];
        $params = [];

        if (!empty($_GET['role'])) {
            $where[] = 'role = ?';
            $params[] = $_GET['role'];
        }

        if (!empty($_GET['status'])) {
            $where[] = 'status = ?';
            $params[] = $_GET['status'];
        }

        if (!empty($_GET['search'])) {
            $where[] = '(name LIKE ? OR email LIKE ? OR employee_id LIKE ?)';
            $search = '%' . $_GET['search'] . '%';
            $params = array_merge($params, [$search, $search, $search]);
        }

        $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

        $users = $this->db->fetchAll(
            "SELECT id, employee_id, name, email, phone, role, department, avatar, status, last_login, created_at 
             FROM users {$whereClause} 
             ORDER BY name ASC 
             LIMIT ? OFFSET ?",
            array_merge($params, [$perPage, $offset])
        );

        $total = $this->db->fetchOne("SELECT COUNT(*) as count FROM users {$whereClause}", $params);

        Response::paginated($users, (int) $total['count'], $page, $perPage);
    }

    private function getUser(int $id): void {
        $user = $this->db->fetchOne(
            "SELECT id, employee_id, name, email, phone, role, department, avatar, status, last_login, created_at 
             FROM users WHERE id = ?",
            [$id]
        );

        if (!$user) {
            Response::notFound('User not found');
        }

        // Get user status
        $status = $this->db->fetchOne(
            "SELECT status, last_activity, current_task_id FROM user_status WHERE user_id = ?",
            [$id]
        );
        $user['live_status'] = $status;

        // Get current task if any
        if ($status && $status['current_task_id']) {
            $user['current_task'] = $this->db->fetchOne(
                "SELECT task_id, title FROM tasks WHERE id = ?",
                [$status['current_task_id']]
            );
        }

        Response::success($user);
    }

    private function getStaffList(): void {
        $users = $this->db->fetchAll(
            "SELECT id, employee_id, name, email, role 
             FROM users 
             WHERE status = 'active' 
             ORDER BY name ASC"
        );

        Response::success($users);
    }

    private function createUser(array $input): void {
        $validator = new Validator($input);
        $validator
            ->required('employee_id')
            ->required('name')
            ->required('email')
            ->email('email')
            ->required('password')
            ->minLength('password', 6)
            ->required('role')
            ->in('role', ['admin', 'manager', 'staff'])
            ->unique('email', 'users')
            ->unique('employee_id', 'users')
            ->validate();

        $userId = $this->db->insert('users', [
            'employee_id' => $input['employee_id'],
            'name' => $input['name'],
            'email' => $input['email'],
            'password' => password_hash($input['password'], PASSWORD_DEFAULT),
            'phone' => $input['phone'] ?? null,
            'role' => $input['role'],
            'department' => $input['department'] ?? null,
            'status' => $input['status'] ?? 'active'
        ]);

        AuditService::log('create', 'user', $userId, null, $input);

        $user = $this->db->fetchOne(
            "SELECT id, employee_id, name, email, role, status FROM users WHERE id = ?",
            [$userId]
        );

        Response::created($user);
    }

    private function updateUser(int $id, array $input): void {
        $currentUser = AuthMiddleware::getCurrentUser();
        
        // Users can update their own profile, admins can update anyone
        if ($currentUser['id'] !== $id && $currentUser['role'] !== 'admin') {
            Response::forbidden('Cannot update other users');
        }

        $user = $this->db->fetchOne("SELECT * FROM users WHERE id = ?", [$id]);
        if (!$user) {
            Response::notFound('User not found');
        }

        $updateData = [];

        if (isset($input['name'])) {
            $updateData['name'] = $input['name'];
        }
        if (isset($input['phone'])) {
            $updateData['phone'] = $input['phone'];
        }
        if (isset($input['department'])) {
            $updateData['department'] = $input['department'];
        }

        // Only admin can update these fields
        if ($currentUser['role'] === 'admin') {
            if (isset($input['role'])) {
                $updateData['role'] = $input['role'];
            }
            if (isset($input['status'])) {
                $updateData['status'] = $input['status'];
            }
            if (isset($input['employee_id']) && $input['employee_id'] !== $user['employee_id']) {
                $validator = new Validator($input);
                $validator->unique('employee_id', 'users', 'employee_id', $id)->validate();
                $updateData['employee_id'] = $input['employee_id'];
            }
        }

        if (empty($updateData)) {
            Response::error('No valid fields to update');
        }

        $this->db->update('users', $updateData, 'id = ?', [$id]);

        AuditService::log('update', 'user', $id, $user, $updateData);

        $updatedUser = $this->db->fetchOne(
            "SELECT id, employee_id, name, email, phone, role, department, status FROM users WHERE id = ?",
            [$id]
        );

        Response::success($updatedUser, 'User updated');
    }

    private function deleteUser(int $id): void {
        $user = $this->db->fetchOne("SELECT * FROM users WHERE id = ?", [$id]);
        
        if (!$user) {
            Response::notFound('User not found');
        }

        if ($user['role'] === 'admin') {
            $adminCount = $this->db->fetchOne("SELECT COUNT(*) as count FROM users WHERE role = 'admin'");
            if ($adminCount['count'] <= 1) {
                Response::error('Cannot delete the last admin user');
            }
        }

        // Soft delete by setting status to inactive
        $this->db->update('users', ['status' => 'inactive'], 'id = ?', [$id]);

        AuditService::log('delete', 'user', $id, $user);

        Response::success(null, 'User deleted');
    }

    private function updateStatus(array $input): void {
        $user = AuthMiddleware::getCurrentUser();

        $validator = new Validator($input);
        $validator
            ->required('status')
            ->in('status', ['online', 'offline', 'busy', 'away', 'on_task'])
            ->validate();

        $this->db->query(
            "INSERT INTO user_status (user_id, status, last_activity, gps_lat, gps_lng) 
             VALUES (?, ?, NOW(), ?, ?) 
             ON DUPLICATE KEY UPDATE status = ?, last_activity = NOW(), gps_lat = ?, gps_lng = ?",
            [
                $user['id'],
                $input['status'],
                $input['gps_lat'] ?? null,
                $input['gps_lng'] ?? null,
                $input['status'],
                $input['gps_lat'] ?? null,
                $input['gps_lng'] ?? null
            ]
        );

        Response::success(null, 'Status updated');
    }
}
