<?php
/**
 * Cron Job - Generate Recurring Tasks
 * Run daily: 0 0 * * * php /path/to/recurring_tasks.php
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/Response.php';
require_once __DIR__ . '/../services/NotificationService.php';

$db = Database::getInstance();
$today = date('Y-m-d');

echo "Running recurring tasks generation for {$today}\n";

$tasks = $db->fetchAll(
    "SELECT * FROM recurring_tasks WHERE is_active = 1 AND (next_run IS NULL OR next_run <= ?)",
    [$today]
);

$created = 0;

foreach ($tasks as $recurringTask) {
    $existing = $db->fetchOne(
        "SELECT id FROM tasks WHERE recurring_task_id = ? AND DATE(created_at) = ?",
        [$recurringTask['id'], $today]
    );

    if ($existing) {
        echo "Task already generated for recurring task ID {$recurringTask['id']}\n";
        continue;
    }

    $db->query("CALL generate_task_id(@new_id)");
    $result = $db->fetchOne("SELECT @new_id as task_id");
    $taskId = $result['task_id'];

    $newTaskId = $db->insert('tasks', [
        'task_id' => $taskId,
        'title' => $recurringTask['title'],
        'description' => $recurringTask['description'],
        'type' => $recurringTask['type'],
        'priority' => $recurringTask['priority'],
        'status' => 'new',
        'created_by' => $recurringTask['created_by'],
        'is_recurring' => 1,
        'recurring_task_id' => $recurringTask['id'],
        'sla_deadline' => date('Y-m-d H:i:s', strtotime('+24 hours'))
    ]);

    if ($recurringTask['assign_to']) {
        $assignees = json_decode($recurringTask['assign_to'], true);
        foreach ($assignees as $index => $assigneeId) {
            $db->insert('task_assignments', [
                'task_id' => $newTaskId,
                'user_id' => $assigneeId,
                'assigned_by' => $recurringTask['created_by'],
                'is_primary' => $index === 0 ? 1 : 0
            ]);
        }
    }

    $frequency = $recurringTask['frequency'];
    $nextRun = new DateTime();
    
    switch ($frequency) {
        case 'daily':
            $nextRun->modify('+1 day');
            break;
        case 'weekly':
            $nextRun->modify('+1 week');
            break;
        case 'monthly':
            $nextRun->modify('+1 month');
            break;
    }

    $db->update('recurring_tasks', [
        'last_generated' => $today,
        'next_run' => $nextRun->format('Y-m-d')
    ], 'id = ?', [$recurringTask['id']]);

    echo "Created task {$taskId} from recurring task ID {$recurringTask['id']}\n";
    $created++;
}

echo "Completed. Created {$created} tasks.\n";
