-- IT Task Workflow & Attendance Management System
-- Database Schema

SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET AUTOCOMMIT = 0;
START TRANSACTION;
SET time_zone = "+00:00";

-- --------------------------------------------------------
-- Database: `it_tasks_db`
-- --------------------------------------------------------

CREATE DATABASE IF NOT EXISTS `it_tasks_db` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE `it_tasks_db`;

-- --------------------------------------------------------
-- Table: users
-- --------------------------------------------------------
CREATE TABLE `users` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `employee_id` VARCHAR(50) UNIQUE NOT NULL,
    `name` VARCHAR(100) NOT NULL,
    `email` VARCHAR(150) UNIQUE NOT NULL,
    `password` VARCHAR(255) NOT NULL,
    `phone` VARCHAR(20) NULL,
    `role` ENUM('admin', 'manager', 'staff') NOT NULL DEFAULT 'staff',
    `department` VARCHAR(100) NULL,
    `avatar` VARCHAR(255) NULL,
    `status` ENUM('active', 'inactive', 'suspended') NOT NULL DEFAULT 'active',
    `last_login` DATETIME NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_users_role` (`role`),
    INDEX `idx_users_status` (`status`),
    INDEX `idx_users_email` (`email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: device_tokens (for push notifications)
-- --------------------------------------------------------
CREATE TABLE `device_tokens` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `token` VARCHAR(500) NOT NULL,
    `device_type` ENUM('android', 'ios', 'web') NOT NULL,
    `device_name` VARCHAR(100) NULL,
    `is_active` TINYINT(1) DEFAULT 1,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    UNIQUE KEY `unique_token` (`token`(255)),
    INDEX `idx_device_user` (`user_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: tasks
-- --------------------------------------------------------
CREATE TABLE `tasks` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `task_id` VARCHAR(20) UNIQUE NOT NULL COMMENT 'Format: IT-YEAR-0001',
    `title` VARCHAR(255) NOT NULL,
    `description` TEXT NULL,
    `type` ENUM('support', 'maintenance', 'visit', 'internal', 'urgent') NOT NULL DEFAULT 'support',
    `priority` ENUM('low', 'medium', 'high', 'urgent') NOT NULL DEFAULT 'medium',
    `status` ENUM('new', 'started', 'on_hold', 'completed', 'cancelled') NOT NULL DEFAULT 'new',
    `created_by` INT UNSIGNED NOT NULL,
    `start_time` DATETIME NULL,
    `end_time` DATETIME NULL,
    `duration_minutes` INT UNSIGNED NULL COMMENT 'Auto calculated',
    `hold_reason` TEXT NULL,
    `gps_lat_start` DECIMAL(10, 8) NULL,
    `gps_lng_start` DECIMAL(11, 8) NULL,
    `gps_lat_end` DECIMAL(10, 8) NULL,
    `gps_lng_end` DECIMAL(11, 8) NULL,
    `location_name` VARCHAR(255) NULL,
    `sla_deadline` DATETIME NULL,
    `sla_breached` TINYINT(1) DEFAULT 0,
    `is_recurring` TINYINT(1) DEFAULT 0,
    `recurring_task_id` INT UNSIGNED NULL,
    `template_id` INT UNSIGNED NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`created_by`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
    INDEX `idx_tasks_status` (`status`),
    INDEX `idx_tasks_type` (`type`),
    INDEX `idx_tasks_priority` (`priority`),
    INDEX `idx_tasks_created_by` (`created_by`),
    INDEX `idx_tasks_task_id` (`task_id`),
    INDEX `idx_tasks_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: task_assignments
-- --------------------------------------------------------
CREATE TABLE `task_assignments` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `task_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `assigned_by` INT UNSIGNED NOT NULL,
    `assigned_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `is_primary` TINYINT(1) DEFAULT 0,
    FOREIGN KEY (`task_id`) REFERENCES `tasks`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`assigned_by`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
    UNIQUE KEY `unique_assignment` (`task_id`, `user_id`),
    INDEX `idx_assignment_user` (`user_id`),
    INDEX `idx_assignment_task` (`task_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: task_logs (audit trail)
-- --------------------------------------------------------
CREATE TABLE `task_logs` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `task_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `action` VARCHAR(50) NOT NULL COMMENT 'created, started, paused, resumed, completed, etc.',
    `old_value` TEXT NULL,
    `new_value` TEXT NULL,
    `notes` TEXT NULL,
    `gps_lat` DECIMAL(10, 8) NULL,
    `gps_lng` DECIMAL(11, 8) NULL,
    `ip_address` VARCHAR(45) NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`task_id`) REFERENCES `tasks`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
    INDEX `idx_tasklog_task` (`task_id`),
    INDEX `idx_tasklog_user` (`user_id`),
    INDEX `idx_tasklog_action` (`action`),
    INDEX `idx_tasklog_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: task_attachments
-- --------------------------------------------------------
CREATE TABLE `task_attachments` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `task_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `file_type` ENUM('image', 'document', 'voice', 'video', 'other') NOT NULL,
    `file_name` VARCHAR(255) NOT NULL,
    `file_path` VARCHAR(500) NOT NULL,
    `file_size` INT UNSIGNED NULL,
    `mime_type` VARCHAR(100) NULL,
    `is_evidence` TINYINT(1) DEFAULT 0,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`task_id`) REFERENCES `tasks`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
    INDEX `idx_attachment_task` (`task_id`),
    INDEX `idx_attachment_type` (`file_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: voice_notes
-- --------------------------------------------------------
CREATE TABLE `voice_notes` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `task_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `file_path` VARCHAR(500) NOT NULL,
    `duration_seconds` INT UNSIGNED NULL,
    `transcription` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`task_id`) REFERENCES `tasks`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
    INDEX `idx_voice_task` (`task_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: notifications
-- --------------------------------------------------------
CREATE TABLE `notifications` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `title` VARCHAR(255) NOT NULL,
    `body` TEXT NOT NULL,
    `type` VARCHAR(50) NOT NULL COMMENT 'task_created, task_assigned, task_started, etc.',
    `reference_type` VARCHAR(50) NULL COMMENT 'task, attendance, etc.',
    `reference_id` INT UNSIGNED NULL,
    `data` JSON NULL,
    `is_read` TINYINT(1) DEFAULT 0,
    `is_pushed` TINYINT(1) DEFAULT 0,
    `read_at` DATETIME NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    INDEX `idx_notification_user` (`user_id`),
    INDEX `idx_notification_read` (`is_read`),
    INDEX `idx_notification_type` (`type`),
    INDEX `idx_notification_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: attendance
-- --------------------------------------------------------
CREATE TABLE `attendance` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `date` DATE NOT NULL,
    `check_in` DATETIME NULL,
    `check_out` DATETIME NULL,
    `check_in_gps_lat` DECIMAL(10, 8) NULL,
    `check_in_gps_lng` DECIMAL(11, 8) NULL,
    `check_out_gps_lat` DECIMAL(10, 8) NULL,
    `check_out_gps_lng` DECIMAL(11, 8) NULL,
    `check_in_ip` VARCHAR(45) NULL,
    `check_out_ip` VARCHAR(45) NULL,
    `total_work_minutes` INT UNSIGNED NULL,
    `total_break_minutes` INT UNSIGNED NULL,
    `status` ENUM('present', 'absent', 'vacation', 'sick_leave', 'home_leave') NOT NULL DEFAULT 'present',
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    UNIQUE KEY `unique_attendance` (`user_id`, `date`),
    INDEX `idx_attendance_user` (`user_id`),
    INDEX `idx_attendance_date` (`date`),
    INDEX `idx_attendance_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: attendance_breaks
-- --------------------------------------------------------
CREATE TABLE `attendance_breaks` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `attendance_id` INT UNSIGNED NOT NULL,
    `break_start` DATETIME NOT NULL,
    `break_end` DATETIME NULL,
    `break_type` ENUM('break', 'prayer', 'lunch', 'other') NOT NULL DEFAULT 'break',
    `duration_minutes` INT UNSIGNED NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`attendance_id`) REFERENCES `attendance`(`id`) ON DELETE CASCADE,
    INDEX `idx_break_attendance` (`attendance_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: recurring_tasks
-- --------------------------------------------------------
CREATE TABLE `recurring_tasks` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `title` VARCHAR(255) NOT NULL,
    `description` TEXT NULL,
    `type` ENUM('support', 'maintenance', 'visit', 'internal', 'urgent') NOT NULL DEFAULT 'maintenance',
    `priority` ENUM('low', 'medium', 'high', 'urgent') NOT NULL DEFAULT 'medium',
    `frequency` ENUM('daily', 'weekly', 'monthly') NOT NULL,
    `day_of_week` TINYINT NULL COMMENT '0=Sunday, 6=Saturday for weekly',
    `day_of_month` TINYINT NULL COMMENT '1-31 for monthly',
    `time_of_day` TIME NULL,
    `assign_to` JSON NULL COMMENT 'Array of user IDs',
    `created_by` INT UNSIGNED NOT NULL,
    `is_active` TINYINT(1) DEFAULT 1,
    `last_generated` DATE NULL,
    `next_run` DATE NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`created_by`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
    INDEX `idx_recurring_active` (`is_active`),
    INDEX `idx_recurring_next` (`next_run`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: task_templates
-- --------------------------------------------------------
CREATE TABLE `task_templates` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `name` VARCHAR(100) NOT NULL,
    `title` VARCHAR(255) NOT NULL,
    `description` TEXT NULL,
    `type` ENUM('support', 'maintenance', 'visit', 'internal', 'urgent') NOT NULL,
    `priority` ENUM('low', 'medium', 'high', 'urgent') NOT NULL DEFAULT 'medium',
    `default_assignees` JSON NULL,
    `sla_hours` INT UNSIGNED NULL,
    `created_by` INT UNSIGNED NOT NULL,
    `is_active` TINYINT(1) DEFAULT 1,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`created_by`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
    INDEX `idx_template_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: time_tracking (for pause/resume)
-- --------------------------------------------------------
CREATE TABLE `time_tracking` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `task_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `action` ENUM('start', 'pause', 'resume', 'stop') NOT NULL,
    `timestamp` DATETIME NOT NULL,
    `gps_lat` DECIMAL(10, 8) NULL,
    `gps_lng` DECIMAL(11, 8) NULL,
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`task_id`) REFERENCES `tasks`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
    INDEX `idx_tracking_task` (`task_id`),
    INDEX `idx_tracking_user` (`user_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: user_status (live status)
-- --------------------------------------------------------
CREATE TABLE `user_status` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED UNIQUE NOT NULL,
    `status` ENUM('online', 'offline', 'busy', 'away', 'on_task') NOT NULL DEFAULT 'offline',
    `current_task_id` INT UNSIGNED NULL,
    `last_activity` DATETIME NULL,
    `gps_lat` DECIMAL(10, 8) NULL,
    `gps_lng` DECIMAL(11, 8) NULL,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`current_task_id`) REFERENCES `tasks`(`id`) ON DELETE SET NULL,
    INDEX `idx_userstatus_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: refresh_tokens
-- --------------------------------------------------------
CREATE TABLE `refresh_tokens` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `token` VARCHAR(500) NOT NULL,
    `expires_at` DATETIME NOT NULL,
    `is_revoked` TINYINT(1) DEFAULT 0,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    UNIQUE KEY `unique_refresh_token` (`token`(255)),
    INDEX `idx_refresh_user` (`user_id`),
    INDEX `idx_refresh_expires` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: audit_logs (system-wide)
-- --------------------------------------------------------
CREATE TABLE `audit_logs` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NULL,
    `action` VARCHAR(100) NOT NULL,
    `entity_type` VARCHAR(50) NOT NULL,
    `entity_id` INT UNSIGNED NULL,
    `old_data` JSON NULL,
    `new_data` JSON NULL,
    `ip_address` VARCHAR(45) NULL,
    `user_agent` VARCHAR(500) NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE SET NULL,
    INDEX `idx_audit_user` (`user_id`),
    INDEX `idx_audit_entity` (`entity_type`, `entity_id`),
    INDEX `idx_audit_action` (`action`),
    INDEX `idx_audit_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: settings
-- --------------------------------------------------------
CREATE TABLE `settings` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `key` VARCHAR(100) UNIQUE NOT NULL,
    `value` TEXT NULL,
    `type` ENUM('string', 'integer', 'boolean', 'json') NOT NULL DEFAULT 'string',
    `description` VARCHAR(255) NULL,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Insert default settings
-- --------------------------------------------------------
INSERT INTO `settings` (`key`, `value`, `type`, `description`) VALUES
('app_name', 'IT Task Manager', 'string', 'Application name'),
('sla_default_hours', '24', 'integer', 'Default SLA hours for tasks'),
('task_id_prefix', 'IT', 'string', 'Task ID prefix'),
('work_start_time', '08:00', 'string', 'Work start time'),
('work_end_time', '17:00', 'string', 'Work end time'),
('allow_overtime', 'true', 'boolean', 'Allow overtime tracking'),
('max_file_size_mb', '10', 'integer', 'Maximum file upload size in MB'),
('firebase_enabled', 'true', 'boolean', 'Enable Firebase push notifications');

-- --------------------------------------------------------
-- Insert default admin user (password: admin123)
-- --------------------------------------------------------
INSERT INTO `users` (`employee_id`, `name`, `email`, `password`, `role`, `status`) VALUES
('EMP001', 'System Admin', 'admin@ittasks.local', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin', 'active');

-- --------------------------------------------------------
-- Create task ID sequence helper
-- --------------------------------------------------------
CREATE TABLE `task_sequence` (
    `year` YEAR PRIMARY KEY,
    `last_number` INT UNSIGNED DEFAULT 0
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `task_sequence` (`year`, `last_number`) VALUES (YEAR(CURDATE()), 0);

-- --------------------------------------------------------
-- Stored procedure: Generate Task ID
-- --------------------------------------------------------
DELIMITER //
CREATE PROCEDURE `generate_task_id`(OUT new_task_id VARCHAR(20))
BEGIN
    DECLARE current_year YEAR;
    DECLARE next_number INT;
    DECLARE prefix VARCHAR(10);
    
    SET current_year = YEAR(CURDATE());
    
    SELECT `value` INTO prefix FROM `settings` WHERE `key` = 'task_id_prefix';
    IF prefix IS NULL THEN SET prefix = 'IT'; END IF;
    
    INSERT INTO `task_sequence` (`year`, `last_number`) 
    VALUES (current_year, 1)
    ON DUPLICATE KEY UPDATE `last_number` = `last_number` + 1;
    
    SELECT `last_number` INTO next_number FROM `task_sequence` WHERE `year` = current_year;
    
    SET new_task_id = CONCAT(prefix, '-', current_year, '-', LPAD(next_number, 4, '0'));
END //
DELIMITER ;

COMMIT;
