<?php
/**
 * JWT Helper Class
 */

class JWT {
    
    public static function encode(array $payload): string {
        $header = self::base64UrlEncode(json_encode([
            'typ' => 'JWT',
            'alg' => JWT_ALGORITHM
        ]));

        $payload['iat'] = time();
        $payload['exp'] = time() + JWT_ACCESS_EXPIRY;
        
        $payloadEncoded = self::base64UrlEncode(json_encode($payload));
        
        $signature = self::sign("{$header}.{$payloadEncoded}");
        
        return "{$header}.{$payloadEncoded}.{$signature}";
    }

    public static function decode(string $token): ?array {
        $parts = explode('.', $token);
        
        if (count($parts) !== 3) {
            return null;
        }

        [$header, $payload, $signature] = $parts;

        if (!self::verify("{$header}.{$payload}", $signature)) {
            return null;
        }

        $payload = json_decode(self::base64UrlDecode($payload), true);
        
        if (!$payload) {
            return null;
        }

        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return null;
        }

        return $payload;
    }

    public static function generateRefreshToken(): string {
        return bin2hex(random_bytes(64));
    }

    public static function createTokenPair(array $userData): array {
        $accessToken = self::encode($userData);
        $refreshToken = self::generateRefreshToken();
        
        return [
            'access_token' => $accessToken,
            'refresh_token' => $refreshToken,
            'token_type' => 'Bearer',
            'expires_in' => JWT_ACCESS_EXPIRY
        ];
    }

    private static function sign(string $data): string {
        $signature = hash_hmac('sha256', $data, JWT_SECRET, true);
        return self::base64UrlEncode($signature);
    }

    private static function verify(string $data, string $signature): bool {
        $expected = self::sign($data);
        return hash_equals($expected, $signature);
    }

    private static function base64UrlEncode(string $data): string {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    private static function base64UrlDecode(string $data): string {
        $remainder = strlen($data) % 4;
        if ($remainder) {
            $data .= str_repeat('=', 4 - $remainder);
        }
        return base64_decode(strtr($data, '-_', '+/'));
    }

    public static function extractFromHeader(): ?string {
        $headers = getallheaders();
        $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? '';
        
        if (preg_match('/Bearer\s+(.+)$/i', $authHeader, $matches)) {
            return $matches[1];
        }
        
        return null;
    }
}
