<?php
/**
 * Input Validation Helper
 */

class Validator {
    private array $errors = [];
    private array $data;

    public function __construct(array $data) {
        $this->data = $data;
    }

    public function required(string $field, string $message = null): self {
        if (!isset($this->data[$field]) || trim($this->data[$field]) === '') {
            $this->errors[$field] = $message ?? "{$field} is required";
        }
        return $this;
    }

    public function email(string $field, string $message = null): self {
        if (isset($this->data[$field]) && !filter_var($this->data[$field], FILTER_VALIDATE_EMAIL)) {
            $this->errors[$field] = $message ?? "{$field} must be a valid email";
        }
        return $this;
    }

    public function minLength(string $field, int $length, string $message = null): self {
        if (isset($this->data[$field]) && strlen($this->data[$field]) < $length) {
            $this->errors[$field] = $message ?? "{$field} must be at least {$length} characters";
        }
        return $this;
    }

    public function maxLength(string $field, int $length, string $message = null): self {
        if (isset($this->data[$field]) && strlen($this->data[$field]) > $length) {
            $this->errors[$field] = $message ?? "{$field} must not exceed {$length} characters";
        }
        return $this;
    }

    public function numeric(string $field, string $message = null): self {
        if (isset($this->data[$field]) && !is_numeric($this->data[$field])) {
            $this->errors[$field] = $message ?? "{$field} must be numeric";
        }
        return $this;
    }

    public function in(string $field, array $values, string $message = null): self {
        if (isset($this->data[$field]) && !in_array($this->data[$field], $values)) {
            $this->errors[$field] = $message ?? "{$field} must be one of: " . implode(', ', $values);
        }
        return $this;
    }

    public function date(string $field, string $format = 'Y-m-d', string $message = null): self {
        if (isset($this->data[$field])) {
            $date = DateTime::createFromFormat($format, $this->data[$field]);
            if (!$date || $date->format($format) !== $this->data[$field]) {
                $this->errors[$field] = $message ?? "{$field} must be a valid date ({$format})";
            }
        }
        return $this;
    }

    public function datetime(string $field, string $message = null): self {
        return $this->date($field, 'Y-m-d H:i:s', $message);
    }

    public function array(string $field, string $message = null): self {
        if (isset($this->data[$field]) && !is_array($this->data[$field])) {
            $this->errors[$field] = $message ?? "{$field} must be an array";
        }
        return $this;
    }

    public function latitude(string $field, string $message = null): self {
        if (isset($this->data[$field])) {
            $lat = (float) $this->data[$field];
            if ($lat < -90 || $lat > 90) {
                $this->errors[$field] = $message ?? "{$field} must be a valid latitude";
            }
        }
        return $this;
    }

    public function longitude(string $field, string $message = null): self {
        if (isset($this->data[$field])) {
            $lng = (float) $this->data[$field];
            if ($lng < -180 || $lng > 180) {
                $this->errors[$field] = $message ?? "{$field} must be a valid longitude";
            }
        }
        return $this;
    }

    public function unique(string $field, string $table, string $column = null, int $exceptId = null): self {
        if (isset($this->data[$field])) {
            $column = $column ?? $field;
            $db = Database::getInstance();
            
            $sql = "SELECT id FROM {$table} WHERE {$column} = ?";
            $params = [$this->data[$field]];
            
            if ($exceptId !== null) {
                $sql .= " AND id != ?";
                $params[] = $exceptId;
            }
            
            $result = $db->fetchOne($sql, $params);
            if ($result) {
                $this->errors[$field] = "{$field} already exists";
            }
        }
        return $this;
    }

    public function custom(string $field, callable $callback, string $message): self {
        if (isset($this->data[$field]) && !$callback($this->data[$field])) {
            $this->errors[$field] = $message;
        }
        return $this;
    }

    public function isValid(): bool {
        return empty($this->errors);
    }

    public function getErrors(): array {
        return $this->errors;
    }

    public function validate(): void {
        if (!$this->isValid()) {
            Response::validationError($this->errors);
        }
    }

    public static function sanitize(string $value): string {
        return htmlspecialchars(trim($value), ENT_QUOTES, 'UTF-8');
    }

    public static function sanitizeArray(array $data): array {
        return array_map(function ($value) {
            if (is_string($value)) {
                return self::sanitize($value);
            }
            if (is_array($value)) {
                return self::sanitizeArray($value);
            }
            return $value;
        }, $data);
    }
}
