<?php
/**
 * IT Task Management System - API Entry Point
 */

// Load configuration
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/config/database.php';

// Load helpers
require_once __DIR__ . '/helpers/Response.php';
require_once __DIR__ . '/helpers/Validator.php';
require_once __DIR__ . '/helpers/JWT.php';

// Load middleware
require_once __DIR__ . '/middleware/CorsMiddleware.php';
require_once __DIR__ . '/middleware/AuthMiddleware.php';
require_once __DIR__ . '/middleware/RateLimitMiddleware.php';

// Load services
require_once __DIR__ . '/services/NotificationService.php';
require_once __DIR__ . '/services/FileService.php';
require_once __DIR__ . '/services/AuditService.php';

// Handle CORS
CorsMiddleware::handle();

// Get request info
$requestMethod = $_SERVER['REQUEST_METHOD'];
$requestUri = $_SERVER['REQUEST_URI'];

// Parse URI - remove base path and query string
$basePath = '/ittask/api';
$uri = parse_url($requestUri, PHP_URL_PATH);
$uri = str_replace($basePath, '', $uri);
$uri = trim($uri, '/');
$segments = $uri ? explode('/', $uri) : [];

// Get request body
$input = json_decode(file_get_contents('php://input'), true) ?? [];

// Route the request
try {
    // Rate limiting
    RateLimitMiddleware::check();

    $resource = $segments[0] ?? '';
    $id = $segments[1] ?? null;
    $action = $segments[2] ?? null;

    switch ($resource) {
        case 'auth':
            require_once __DIR__ . '/controllers/AuthController.php';
            $controller = new AuthController();
            $controller->handle($requestMethod, $id, $input);
            break;

        case 'users':
            require_once __DIR__ . '/controllers/UserController.php';
            $controller = new UserController();
            $controller->handle($requestMethod, $id, $action, $input);
            break;

        case 'tasks':
            require_once __DIR__ . '/controllers/TaskController.php';
            $controller = new TaskController();
            $controller->handle($requestMethod, $id, $action, $input);
            break;

        case 'attendance':
            require_once __DIR__ . '/controllers/AttendanceController.php';
            $controller = new AttendanceController();
            $controller->handle($requestMethod, $id, $action, $input);
            break;

        case 'notifications':
            require_once __DIR__ . '/controllers/NotificationController.php';
            $controller = new NotificationController();
            $controller->handle($requestMethod, $id, $action, $input);
            break;

        case 'recurring-tasks':
            require_once __DIR__ . '/controllers/RecurringTaskController.php';
            $controller = new RecurringTaskController();
            $controller->handle($requestMethod, $id, $action, $input);
            break;

        case 'templates':
            require_once __DIR__ . '/controllers/TemplateController.php';
            $controller = new TemplateController();
            $controller->handle($requestMethod, $id, $input);
            break;

        case 'dashboard':
            require_once __DIR__ . '/controllers/DashboardController.php';
            $controller = new DashboardController();
            $controller->handle($requestMethod, $action, $input);
            break;

        case 'upload':
            require_once __DIR__ . '/controllers/UploadController.php';
            $controller = new UploadController();
            $controller->handle($requestMethod, $id, $input);
            break;

        case '':
            Response::success([
                'name' => 'IT Task Management API',
                'version' => API_VERSION,
                'status' => 'running'
            ]);
            break;

        default:
            Response::notFound('Endpoint not found');
    }
} catch (PDOException $e) {
    error_log("Database Error: " . $e->getMessage());
    Response::serverError('Database error occurred');
} catch (Exception $e) {
    error_log("Error: " . $e->getMessage());
    Response::error($e->getMessage(), 400);
}
