<?php
/**
 * Authentication Middleware
 */

class AuthMiddleware {
    private static ?array $currentUser = null;

    public static function authenticate(): array {
        $token = JWT::extractFromHeader();
        
        if (!$token) {
            Response::unauthorized('No token provided');
        }

        $payload = JWT::decode($token);
        
        if (!$payload) {
            Response::unauthorized('Invalid or expired token');
        }

        $db = Database::getInstance();
        $user = $db->fetchOne(
            "SELECT id, employee_id, name, email, role, status FROM users WHERE id = ? AND status = 'active'",
            [$payload['user_id']]
        );

        if (!$user) {
            Response::unauthorized('User not found or inactive');
        }

        self::$currentUser = $user;
        return $user;
    }

    public static function requireRole(array $roles): void {
        $user = self::getCurrentUser();
        
        if (!$user) {
            self::authenticate();
            $user = self::getCurrentUser();
        }

        if (!in_array($user['role'], $roles)) {
            Response::forbidden('Insufficient permissions');
        }
    }

    public static function requireAdmin(): void {
        self::requireRole(['admin']);
    }

    public static function requireManager(): void {
        self::requireRole(['admin', 'manager']);
    }

    public static function requireStaff(): void {
        self::requireRole(['admin', 'manager', 'staff']);
    }

    public static function getCurrentUser(): ?array {
        return self::$currentUser;
    }

    public static function getUserId(): ?int {
        return self::$currentUser['id'] ?? null;
    }

    public static function isAdmin(): bool {
        return self::$currentUser && self::$currentUser['role'] === 'admin';
    }

    public static function isManager(): bool {
        return self::$currentUser && in_array(self::$currentUser['role'], ['admin', 'manager']);
    }

    public static function optional(): ?array {
        $token = JWT::extractFromHeader();
        
        if (!$token) {
            return null;
        }

        $payload = JWT::decode($token);
        
        if (!$payload) {
            return null;
        }

        $db = Database::getInstance();
        $user = $db->fetchOne(
            "SELECT id, employee_id, name, email, role, status FROM users WHERE id = ? AND status = 'active'",
            [$payload['user_id']]
        );

        if ($user) {
            self::$currentUser = $user;
        }

        return $user;
    }
}
