<?php
/**
 * Rate Limiting Middleware
 */

class RateLimitMiddleware {
    private static string $cacheDir = __DIR__ . '/../cache/rate_limits/';

    public static function check(string $identifier = null): void {
        if (!is_dir(self::$cacheDir)) {
            mkdir(self::$cacheDir, 0755, true);
        }

        $identifier = $identifier ?? self::getClientIdentifier();
        $cacheFile = self::$cacheDir . md5($identifier) . '.json';
        
        $data = self::loadData($cacheFile);
        $now = time();

        // Clean old entries
        $data = array_filter($data, fn($timestamp) => $timestamp > ($now - RATE_LIMIT_WINDOW));

        if (count($data) >= RATE_LIMIT_REQUESTS) {
            $retryAfter = min($data) + RATE_LIMIT_WINDOW - $now;
            header('Retry-After: ' . $retryAfter);
            Response::error('Rate limit exceeded. Try again later.', 429);
        }

        $data[] = $now;
        self::saveData($cacheFile, $data);
    }

    private static function getClientIdentifier(): string {
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $userId = AuthMiddleware::getUserId();
        
        return $userId ? "user_{$userId}" : "ip_{$ip}";
    }

    private static function loadData(string $file): array {
        if (!file_exists($file)) {
            return [];
        }
        
        $content = file_get_contents($file);
        return json_decode($content, true) ?? [];
    }

    private static function saveData(string $file, array $data): void {
        file_put_contents($file, json_encode($data));
    }

    public static function cleanup(): void {
        $files = glob(self::$cacheDir . '*.json');
        $now = time();

        foreach ($files as $file) {
            if (filemtime($file) < ($now - RATE_LIMIT_WINDOW * 2)) {
                unlink($file);
            }
        }
    }
}
