<?php
/**
 * Audit Logging Service
 */

class AuditService {
    
    public static function log(
        string $action,
        string $entityType,
        ?int $entityId = null,
        ?array $oldData = null,
        ?array $newData = null
    ): void {
        $db = Database::getInstance();
        
        $db->insert('audit_logs', [
            'user_id' => AuthMiddleware::getUserId(),
            'action' => $action,
            'entity_type' => $entityType,
            'entity_id' => $entityId,
            'old_data' => $oldData ? json_encode($oldData) : null,
            'new_data' => $newData ? json_encode($newData) : null,
            'ip_address' => self::getClientIp(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    }

    public static function taskLog(
        int $taskId,
        string $action,
        ?string $oldValue = null,
        ?string $newValue = null,
        ?string $notes = null,
        ?float $gpsLat = null,
        ?float $gpsLng = null
    ): void {
        $db = Database::getInstance();
        
        $db->insert('task_logs', [
            'task_id' => $taskId,
            'user_id' => AuthMiddleware::getUserId(),
            'action' => $action,
            'old_value' => $oldValue,
            'new_value' => $newValue,
            'notes' => $notes,
            'gps_lat' => $gpsLat,
            'gps_lng' => $gpsLng,
            'ip_address' => self::getClientIp()
        ]);
    }

    public static function getTaskLogs(int $taskId): array {
        $db = Database::getInstance();
        
        return $db->fetchAll(
            "SELECT tl.*, u.name as user_name 
             FROM task_logs tl 
             JOIN users u ON tl.user_id = u.id 
             WHERE tl.task_id = ? 
             ORDER BY tl.created_at DESC",
            [$taskId]
        );
    }

    public static function getAuditLogs(array $filters = [], int $page = 1, int $perPage = 50): array {
        $db = Database::getInstance();
        
        $where = [];
        $params = [];

        if (!empty($filters['user_id'])) {
            $where[] = 'al.user_id = ?';
            $params[] = $filters['user_id'];
        }

        if (!empty($filters['entity_type'])) {
            $where[] = 'al.entity_type = ?';
            $params[] = $filters['entity_type'];
        }

        if (!empty($filters['action'])) {
            $where[] = 'al.action = ?';
            $params[] = $filters['action'];
        }

        if (!empty($filters['date_from'])) {
            $where[] = 'al.created_at >= ?';
            $params[] = $filters['date_from'] . ' 00:00:00';
        }

        if (!empty($filters['date_to'])) {
            $where[] = 'al.created_at <= ?';
            $params[] = $filters['date_to'] . ' 23:59:59';
        }

        $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
        $offset = ($page - 1) * $perPage;

        $logs = $db->fetchAll(
            "SELECT al.*, u.name as user_name 
             FROM audit_logs al 
             LEFT JOIN users u ON al.user_id = u.id 
             {$whereClause} 
             ORDER BY al.created_at DESC 
             LIMIT ? OFFSET ?",
            array_merge($params, [$perPage, $offset])
        );

        $total = $db->fetchOne(
            "SELECT COUNT(*) as count FROM audit_logs al {$whereClause}",
            $params
        );

        return [
            'logs' => $logs,
            'total' => (int) $total['count']
        ];
    }

    public static function getClientIp(): string {
        $headers = ['HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'REMOTE_ADDR'];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return 'unknown';
    }
}
