<?php
/**
 * File Upload Service
 */

class FileService {
    
    public static function upload(array $file, string $subDir = 'general'): array {
        self::validateFile($file);

        $uploadDir = UPLOAD_PATH . $subDir . '/' . date('Y/m/');
        
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }

        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $filename = uniqid() . '_' . time() . '.' . $extension;
        $filepath = $uploadDir . $filename;

        if (!move_uploaded_file($file['tmp_name'], $filepath)) {
            throw new Exception('Failed to upload file');
        }

        $relativePath = $subDir . '/' . date('Y/m/') . $filename;

        return [
            'file_name' => $file['name'],
            'file_path' => $relativePath,
            'file_size' => $file['size'],
            'mime_type' => $file['type'],
            'file_type' => self::getFileType($file['type'])
        ];
    }

    public static function uploadMultiple(array $files, string $subDir = 'general'): array {
        $uploaded = [];
        
        foreach ($files['name'] as $index => $name) {
            $file = [
                'name' => $files['name'][$index],
                'type' => $files['type'][$index],
                'tmp_name' => $files['tmp_name'][$index],
                'error' => $files['error'][$index],
                'size' => $files['size'][$index]
            ];
            
            if ($file['error'] === UPLOAD_ERR_OK) {
                $uploaded[] = self::upload($file, $subDir);
            }
        }

        return $uploaded;
    }

    public static function uploadBase64(string $base64Data, string $subDir = 'general', string $extension = 'jpg'): array {
        if (preg_match('/^data:([^;]+);base64,/', $base64Data, $matches)) {
            $mimeType = $matches[1];
            $base64Data = preg_replace('/^data:[^;]+;base64,/', '', $base64Data);
            $extension = self::getExtensionFromMime($mimeType);
        }

        $data = base64_decode($base64Data);
        
        if ($data === false) {
            throw new Exception('Invalid base64 data');
        }

        $uploadDir = UPLOAD_PATH . $subDir . '/' . date('Y/m/');
        
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }

        $filename = uniqid() . '_' . time() . '.' . $extension;
        $filepath = $uploadDir . $filename;

        if (file_put_contents($filepath, $data) === false) {
            throw new Exception('Failed to save file');
        }

        $relativePath = $subDir . '/' . date('Y/m/') . $filename;
        $mimeType = $mimeType ?? mime_content_type($filepath);

        return [
            'file_name' => $filename,
            'file_path' => $relativePath,
            'file_size' => strlen($data),
            'mime_type' => $mimeType,
            'file_type' => self::getFileType($mimeType)
        ];
    }

    public static function delete(string $relativePath): bool {
        $filepath = UPLOAD_PATH . $relativePath;
        
        if (file_exists($filepath)) {
            return unlink($filepath);
        }
        
        return false;
    }

    public static function getUrl(string $relativePath): string {
        return API_BASE_URL . '/uploads/' . $relativePath;
    }

    private static function validateFile(array $file): void {
        if ($file['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('File upload error: ' . $file['error']);
        }

        if ($file['size'] > MAX_FILE_SIZE) {
            throw new Exception('File size exceeds maximum allowed size');
        }

        $allowedTypes = array_merge(
            ALLOWED_IMAGE_TYPES,
            ALLOWED_DOC_TYPES,
            ALLOWED_AUDIO_TYPES
        );

        if (!in_array($file['type'], $allowedTypes)) {
            throw new Exception('File type not allowed');
        }
    }

    private static function getFileType(string $mimeType): string {
        if (in_array($mimeType, ALLOWED_IMAGE_TYPES)) {
            return 'image';
        }
        if (in_array($mimeType, ALLOWED_DOC_TYPES)) {
            return 'document';
        }
        if (in_array($mimeType, ALLOWED_AUDIO_TYPES)) {
            return 'voice';
        }
        if (strpos($mimeType, 'video/') === 0) {
            return 'video';
        }
        return 'other';
    }

    private static function getExtensionFromMime(string $mimeType): string {
        $map = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'audio/mpeg' => 'mp3',
            'audio/wav' => 'wav',
            'audio/ogg' => 'ogg',
            'audio/m4a' => 'm4a',
            'audio/x-m4a' => 'm4a',
            'application/pdf' => 'pdf'
        ];

        return $map[$mimeType] ?? 'bin';
    }
}
