<?php
/**
 * Notification Service - Firebase Push + In-App Notifications
 */

class NotificationService {
    
    public static function send(int $userId, string $title, string $body, string $type, array $data = []): int {
        $db = Database::getInstance();
        
        // Create in-app notification
        $notificationId = $db->insert('notifications', [
            'user_id' => $userId,
            'title' => $title,
            'body' => $body,
            'type' => $type,
            'reference_type' => $data['reference_type'] ?? null,
            'reference_id' => $data['reference_id'] ?? null,
            'data' => json_encode($data),
            'is_read' => 0,
            'is_pushed' => 0
        ]);

        // Send push notification
        self::sendPush($userId, $title, $body, $data);

        // Update notification as pushed
        $db->update('notifications', ['is_pushed' => 1], 'id = ?', [$notificationId]);

        return $notificationId;
    }

    public static function sendToMultiple(array $userIds, string $title, string $body, string $type, array $data = []): void {
        foreach ($userIds as $userId) {
            self::send($userId, $title, $body, $type, $data);
        }
    }

    public static function sendToRole(string $role, string $title, string $body, string $type, array $data = []): void {
        $db = Database::getInstance();
        $users = $db->fetchAll("SELECT id FROM users WHERE role = ? AND status = 'active'", [$role]);
        
        foreach ($users as $user) {
            self::send($user['id'], $title, $body, $type, $data);
        }
    }

    public static function sendToManagers(string $title, string $body, string $type, array $data = []): void {
        $db = Database::getInstance();
        $users = $db->fetchAll("SELECT id FROM users WHERE role IN ('admin', 'manager') AND status = 'active'");
        
        foreach ($users as $user) {
            self::send($user['id'], $title, $body, $type, $data);
        }
    }

    private static function sendPush(int $userId, string $title, string $body, array $data = []): bool {
        if (FIREBASE_SERVER_KEY === 'YOUR_FIREBASE_SERVER_KEY') {
            return false;
        }

        $db = Database::getInstance();
        $tokens = $db->fetchAll(
            "SELECT token FROM device_tokens WHERE user_id = ? AND is_active = 1",
            [$userId]
        );

        if (empty($tokens)) {
            return false;
        }

        $tokenList = array_column($tokens, 'token');

        $payload = [
            'registration_ids' => $tokenList,
            'notification' => [
                'title' => $title,
                'body' => $body,
                'sound' => 'default',
                'badge' => self::getUnreadCount($userId)
            ],
            'data' => $data,
            'priority' => 'high'
        ];

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => FIREBASE_API_URL,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: key=' . FIREBASE_SERVER_KEY,
                'Content-Type: application/json'
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POSTFIELDS => json_encode($payload)
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return $httpCode === 200;
    }

    public static function getUnreadCount(int $userId): int {
        $db = Database::getInstance();
        $result = $db->fetchOne(
            "SELECT COUNT(*) as count FROM notifications WHERE user_id = ? AND is_read = 0",
            [$userId]
        );
        return (int) ($result['count'] ?? 0);
    }

    public static function markAsRead(int $notificationId, int $userId): bool {
        $db = Database::getInstance();
        $affected = $db->update(
            'notifications',
            ['is_read' => 1, 'read_at' => date('Y-m-d H:i:s')],
            'id = ? AND user_id = ?',
            [$notificationId, $userId]
        );
        return $affected > 0;
    }

    public static function markAllAsRead(int $userId): int {
        $db = Database::getInstance();
        return $db->update(
            'notifications',
            ['is_read' => 1, 'read_at' => date('Y-m-d H:i:s')],
            'user_id = ? AND is_read = 0',
            [$userId]
        );
    }

    public static function getUserNotifications(int $userId, int $page = 1, int $perPage = 20): array {
        $db = Database::getInstance();
        $offset = ($page - 1) * $perPage;

        $notifications = $db->fetchAll(
            "SELECT * FROM notifications WHERE user_id = ? ORDER BY created_at DESC LIMIT ? OFFSET ?",
            [$userId, $perPage, $offset]
        );

        $total = $db->fetchOne(
            "SELECT COUNT(*) as count FROM notifications WHERE user_id = ?",
            [$userId]
        );

        return [
            'notifications' => $notifications,
            'total' => (int) $total['count'],
            'unread' => self::getUnreadCount($userId)
        ];
    }

    // Task-specific notification helpers
    public static function taskCreated(array $task, int $createdBy): void {
        self::sendToManagers(
            'New Task Created',
            "Task {$task['task_id']}: {$task['title']}",
            'task_created',
            [
                'reference_type' => 'task',
                'reference_id' => $task['id'],
                'task_id' => $task['task_id']
            ]
        );
    }

    public static function taskAssigned(array $task, int $assignedTo): void {
        self::send(
            $assignedTo,
            'New Task Assigned',
            "You have been assigned to: {$task['title']}",
            'task_assigned',
            [
                'reference_type' => 'task',
                'reference_id' => $task['id'],
                'task_id' => $task['task_id']
            ]
        );
    }

    public static function taskStarted(array $task): void {
        self::sendToManagers(
            'Task Started',
            "Task {$task['task_id']} has been started",
            'task_started',
            [
                'reference_type' => 'task',
                'reference_id' => $task['id'],
                'task_id' => $task['task_id']
            ]
        );
    }

    public static function taskCompleted(array $task): void {
        self::sendToManagers(
            'Task Completed',
            "Task {$task['task_id']} has been completed",
            'task_completed',
            [
                'reference_type' => 'task',
                'reference_id' => $task['id'],
                'task_id' => $task['task_id']
            ]
        );
    }
}
